<?php

namespace Drupal\sgd_import_export\Form;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\sgd_dashboard\Services\SiteGuardianCronService;

use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a Website import form.
 */
class ImportWebsitesForm extends FormBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected readonly Connection $database;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected readonly EntityTypeManagerInterface $entityTypeManager;

  /**
   * The file system.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected readonly FileSystemInterface $fileSystem;

  /**
   * The Site Guardian cron service.
   *
   * @var \Drupal\sgd_dashboard\Services\SiteGuardianCronService
   */
  protected readonly SiteGuardianCronService $cronService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {

    $instance = parent::create($container);

    $instance->database = $container->get('database');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->fileSystem = $container->get('file_system');
    $instance->cronService = $container->get('siteguardian.CronService');
    $instance->setMessenger($container->get('messenger'));

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'website_import_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {

    $form['website_config_file'] = [
      '#type' => 'file',
      '#title' => $this->t('Website configuration file'),
      '#description' => $this->t('Allowed types: @extensions.', ['@extensions' => 'yml yaml']),
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Import'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {

    $files = $this->getRequest()->files->get('files', []);

    if (!empty($files['website_config_file'])) {

      $fileUpload = $files['website_config_file'];

      if ($fileUpload->isValid()) {
        $form_state->setValue('website_config_file', $fileUpload->getRealPath());
        return;
      }
    }

    $form_state->setErrorByName('website_config_file', $this->t('The file could not be uploaded.'));
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {

    if ($path = $form_state->getValue('website_config_file')) {

      // Load the file contents.
      $yml = file_get_contents($path);

      // Decode the yaml.
      $websites = YAML::decode($yml);

      // Dont need the file anymore.
      $this->fileSystem->unlink($path);

      // Process the websites.
      foreach ($websites as $website) {

        // Check the client exists and if not create it.
        if (!($clientId = $this->clientExists($website['client']))) {
          $clientId = $this->createClient($website['client']);
        }

        // Now create the website.
        $this->createWebsite($clientId, $website);
      }

      $this->messenger->addMessage($this->t('Imported @count websites.', ['@count' => count($websites)]));
    }
  }

  /**
   * Check if the client node specified exists.
   */
  private function clientExists($clientName): ?int {

    $query = $this->entityTypeManager->getStorage('node')->getQuery();
    $query->condition('title', $clientName);
    $query->accessCheck(FALSE);

    $nids = $query->execute();

    if (!(empty($nids))) {
      return reset($nids);
    }

    return NULL;
  }

  /**
   * Create a client node.
   */
  private function createClient($clientName): ?int {

    // Create a client node.
    $clientNode = $this->entityTypeManager->getStorage('node')->create([
      'type' => 'client',
      'title' => $clientName,
      'status' => 1,
    ]);

    $clientNode->save();

    return $clientNode->id();
  }

  /**
   * Create a website node.
   */
  private function createWebsite($clientId, $websiteData) : void {

    $environmentTypeTermId = $this->getEnvironmentTermId($websiteData['environment_type']);

    // Create a website node.
    $websiteNode = $this->entityTypeManager->getStorage('node')->create([
      'type' => 'website',
      'title' => $websiteData['name'],
      'field_client' => ['target_id' => $clientId],
      'field_environment_type' => ['target_id' => $environmentTypeTermId],
      'field_title_suffix' => $websiteData['title_suffix'],
      'field_http_auth_user' => !empty($websiteData['http_auth_user']) ? $websiteData['http_auth_user'] : NULL,
      'field_http_auth_password' => !empty($websiteData['http_auth_password']) ? $websiteData['http_auth_password'] : NULL,
      'field_url' => $websiteData['url'],
      'field_site_guardian_key' => !empty($websiteData['site_guardian_key']) ? $websiteData['site_guardian_key'] : NULL,
      'field_site_last_checked' => 0,
      'status' => 1,
    ]);

    $websiteNode->save();

    // If crontab is specified then update the websites cron tab with it.
    if (!empty($websiteData['crontab'])) {
      $this->cronService->setCrontab($websiteNode, $websiteData['crontab']);
    }

  }

  /**
   * Get the term ID for the environment specified.
   */
  private function getEnvironmentTermId($termName) : ?int {

    $terms = $this->entityTypeManager->getStorage('taxonomy_term')->loadByProperties([
      'vid' => 'environment_type',
      'name' => $termName,
    ]);

    if (!(empty($terms))) {

      $term = reset($terms);

      return $term->id();
    }

    return NULL;
  }

}
