<?php

namespace Drupal\sgd_dashboard\Controller\SgdCompanion;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\sgd_dashboard\Entity\Bundle\WebsiteBundle;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides route responses for the Watchdog Summary page.
 */
class SgdWatchdogSummaryController extends ControllerBase {

  /**
   * The SGD data service.
   *
   * @var \Drupal\sgd_dashboard\Services\SiteGuardianDataService
   */
  protected $dataService;

  /**
   * The Site Guardian Dashboard Companion plugin manager.
   *
   * @var \Drupal\sdg_dashboard\SgdCompanionPluginManager
   */
  protected $pluginManager;

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container) {

    $instance = parent::create($container);

    $instance->dataService = $container->get('siteguardian.DataService');
    $instance->pluginManager = $container->get('plugin.manager.sgd_companion');

    return $instance;
  }

  /**
   * Display the User data for the website.
   *
   * @return array
   *   A render array as expected by
   *   \Drupal\Core\Render\RendererInterface::render().
   */
  public function view(WebsiteBundle $node, RouteMatchInterface $route_match) {

    // Get the php companion plugin.
    $plugin = $this->pluginManager->createInstance('sgd_companion_watchdog_summary', []);

    /** @var \Drupal\sgd_dashboard\Entity\SgdWebsiteData $websiteData */
    $websiteData = $node->getDataEntity();

    if ($elements = $plugin->getBuildElements($websiteData)) {

      // If the dblog module is installed.
      if ($elements['sgd_watchdog_status']['value'] == 'Yes') {

        $rows = [];

        foreach ($elements['sgd_watchdog_summary'] as $value) {

          $rows[] = [
            'data' => [
              ['data' => $value['type']],
              ['data' => $value['emergency']],
              ['data' => $value['alert']],
              ['data' => $value['critical']],
              ['data' => $value['error']],
              ['data' => $value['warning']],
              ['data' => $value['notice']],
              ['data' => $value['info']],
              ['data' => $value['debug']],
              ['data' => $value['total']],
            ],
            'class' => 'my-row-class',
          ];

        }

        $table = [
          '#theme' => 'table',
          '#caption' => 'Counts of entries by type and severity',
          '#attributes' => ['class' => 'watchdog-summary-table'],
          '#header' => [
            ['data' => 'Type'],
            ['data' => 'Emergency'],
            ['data' => 'Alert'],
            ['data' => 'Critical'],
            ['data' => 'Error'],
            ['data' => 'Warning'],
            ['data' => 'Notice'],
            ['data' => 'Info'],
            ['data' => 'Debug'],
            ['data' => 'Total'],
          ],
          '#rows' => $rows,
        ];

        $build = [
          '#theme' => 'sgd_watchdog_summary',
          '#data' => [
            'sgd_watchdog_summary' => $table,
            'sgd_watchdog_max' => $elements['sgd_watchdog_max'],
            'sgd_watchdog_count' => $elements['sgd_watchdog_count'],
            'sgd_watchdog_oldest' => $elements['sgd_watchdog_oldest'],
          ],
        ];

      }

      else {
        $build = [
          '#markup' => '<p>' . $this->t("No watchdog summary available. Check the 'Database Logging (dblog)' module is installed and enabled on the target website.") . '</p>',
        ];
      }

    }
    else {
      $build = [
        '#markup' => '<p>' . $this->t("No watchdog summary available. Check the 'Site Guardian API Watchdog Summary' companion module is installed and enabled on the target website.") . '</p>',
      ];
    }

    $build['#cache'] = ['max-age' => 0];

    $build['#attached']['library'][] = 'sgd_dashboard/sgd_watchdog_summary';

    return $build;
  }

}
