<?php

namespace Drupal\sgd_dashboard\Plugin\SgdCompanion;

use Drupal\sgd_dashboard\SgdCompanionPluginBase;

/**
 * Provides a SGD Companion plugin.
 *
 * @SgdCompanion(
 *   id = "sgd_companion_watchdog_summary",
 * )
 */
class SgdCompanionWatchdogSummary extends SgdCompanionPluginBase {

  /**
   * {@inheritdoc}
   */
  public function canProcessStatus($statusData) : bool {

    // If the Site Guardian API Companion module 'Watchdog summary' is
    // installed on the site we should have data from it.
    // Just checking for the existance of 'sgd_watchdog_summary' is enough.
    if (array_key_exists('sgd_watchdog_summary', $statusData)) {
      return TRUE;
    }
    else {
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function saveStatus($websiteData, $statusData, $enabledProjects = NULL) : bool {

    if ($this->canProcessStatus($statusData)) {

      $data = [
        'sgd_watchdog_summary' => $statusData['sgd_watchdog_summary'],
      ];

      $websiteData->set('data_watchdog_summary', serialize($data));

    }
    else {
      $websiteData->set('data_watchdog_summary', NULL);
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getStatusDefaults() : array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getStatus($websiteData) : array | NULL {

    if ($dataSerialized = $websiteData->get('data_watchdog_summary')->value) {

      $data = unserialize($dataSerialized, ['allowed_classes' => FALSE]);

      // If the dblog module is enabled on the site.
      if ($data['sgd_watchdog_summary']['enabled']) {

        // Here is the opertunity to massage/transform any data we got from the
        // website status.
        // Do it here rather than when saving so we retain all info received.
        $data['sgd_watchdog_status'] = [
          'title' => "DBLog module enabled.",
          'description' => "Is the core DBLog module enabled.",
          'value' => TRUE,
        ];

        $data['sgd_watchdog_max'] = [
          'title' => "Maximum watchdog entries.",
          'description' => 'The maximum number of watchdog entries retained by the system (admin->configuration->development->logging).',
          'value' => $data['sgd_watchdog_summary']['max'],
        ];

        $data['sgd_watchdog_count'] = [
          'title' => 'Entries in watchdog',
          'description' => 'The number of entries in the watchdog log.',
          'value' => $data['sgd_watchdog_summary']['count'],
        ];

        $data['sgd_watchdog_oldest'] = [
          'title' => 'Oldest entry',
          'description' => 'The date of the oldest entry in the watchdog log.',
          'value' => $data['sgd_watchdog_summary']['oldest'],
        ];

        $data['sgd_watchdog_total_emergency'] = [
          'description' => "Number of entries of type 'emergency'",
          'value' => 0,
        ];

        $data['sgd_watchdog_total_alert'] = [
          'description' => "Number of entries of type 'alert'",
          'value' => 0,
        ];

        $data['sgd_watchdog_total_critical'] = [
          'description' => "Number of entries of type 'critical'",
          'value' => 0,
        ];

        $data['sgd_watchdog_total_error'] = [
          'description' => "Number of entries of type 'error'",
          'value' => 0,
        ];

        $data['sgd_watchdog_total_warning'] = [
          'description' => "Number of entries of type 'warning'",
          'value' => 0,
        ];

        $data['sgd_watchdog_total_notice'] = [
          'description' => "Number of entries of type 'notice'",
          'value' => 0,
        ];

        $data['sgd_watchdog_total_info'] = [
          'description' => "Number of entries of type 'info'",
          'value' => 0,
        ];

        $data['sgd_watchdog_total_debug'] = [
          'description' => "Number of entries of type 'debug'",
          'value' => 0,
        ];

        foreach ($data['sgd_watchdog_summary']['raw'] as $counts) {
          foreach ($counts as $severity => $count) {
            if ($severity != 'type' && $severity != 'total') {
              $data['sgd_watchdog_total_' . $severity]['value'] = empty($data['sgd_watchdog_total_' . $severity]['value']) ? $count : $data['sgd_watchdog_total_' . $severity]['value'] + $count;
            }
          }
        }

        $data['sgd_watchdog_summary'] = $data['sgd_watchdog_summary']['raw'];
      }

      else {
        $data['sgd_watchdog_status'] = [
          'description' => "Is the DBLog module installed.",
          'value' => FALSE,
        ];
      }

    }
    else {
      $data = NULL;
    }

    return $data;
  }

  /**
   * {@inheritdoc}
   */
  public function getBuildElements($websiteData) : array | NULL {

    if ($data = $this->getStatus($websiteData)) {

      $elements = [];

      $validation = $this->validate($data);

      $elements['sgd_watchdog_status'] = [
        'title' => $data['sgd_watchdog_status']['description'],
        'value' => $data['sgd_watchdog_status']['value'] ? 'Yes' : 'No',
        'status' => $validation['sgd_watchdog_status'] ?? NULL,
      ];

      foreach ($data as $key => $value) {
        if ($key != 'sgd_watchdog_summary' && $key != 'sgd_watchdog_status') {
          $elements[$key] = [
            'title' => $value['title'] ?? $value['description'],
            'value' => $value['value'],
            'status' => $validation[$key] ?? NULL,
            'description' => $value['description'],
          ];
        }
      }

      $elements['sgd_watchdog_summary'] = $data['sgd_watchdog_summary'];

      return $elements;
    }

    return NULL;
  }

  /**
   * Validate data.
   *
   * Checks each item and returns a good/nuetral/bad status that can be
   * displayed on page or in a report.
   */
  private function validate(&$statusData): array {

    $validation = [];

    // Validate User variables
    // Each validation is hard coded as it differs for each plugin.
    foreach ($statusData as $key => $status) {

      switch ($key) {

        case 'sgd_watchdog_status':

          if (!$status['value']) {
            $validation[$key] = [
              'class' => 'warning',
              'text' => $this->t('Alert'),
              'message' => $this->t("The 'Database Logging' core module is not enabled. Consider enabling it if you have no other logging in place."),
            ];
          }
          else {
            $validation[$key] = [
              'class' => 'ok',
              'text' => $this->t('OK'),
            ];
          }

          break;

        case 'sgd_watchdog_max':

          if ($status['value'] < 10000) {
            $validation[$key] = [
              'class' => 'warning',
              'text' => $this->t('Alert'),
              'message' => $this->t('The maximum number of watchdog entries retained is &lt; 10000. Consider increasing the value.'),
            ];
          }
          else {
            $validation[$key] = [
              'class' => 'ok',
              'text' => $this->t('OK'),
            ];
          }

          break;

        case 'sgd_watchdog_oldest':

          // If the log is full.
          if ($statusData['sgd_watchdog_count']['value'] >= $statusData['sgd_watchdog_max']['value']) {

            $oldest = new \DateTime();
            $oldest->setTimestamp($status['value']);

            $now = new \DateTime();
            $oldestInDaysAgo = $now->diff($oldest);

            // If the last message is less than 5 days ago.
            if ($oldestInDaysAgo->days <= 5) {
              $validation[$key] = [
                'class' => 'warning',
                'text' => $this->t('Alert'),
                'message' => $this->t('Watchdog is rolling over within 5 days. Reduce entries being created or increase the log size.'),
              ];
            }
            else {
              $validation[$key] = [
                'class' => 'ok',
                'text' => $this->t('OK'),
              ];
            }
          }

          break;

        case 'sgd_watchdog_total_emergency':
        case 'sgd_watchdog_total_alert':
        case 'sgd_watchdog_total_critical':
        case 'sgd_watchdog_total_error':

          if ($status['value'] > 0) {
            $validation[$key] = [
              'class' => 'error',
              'text' => $this->t('Issue'),
              'message' => $this->t('Investigate the cause of the watchdog entries.'),
            ];
          }
          else {
            $validation[$key] = [
              'class' => 'ok',
              'text' => $this->t('OK'),
            ];
          }

          break;

        case 'sgd_watchdog_total_warning':

          if ($status['value'] > 0) {
            $validation[$key] = [
              'class' => 'warning',
              'text' => $this->t('Alert'),
              'message' => $this->t('Investigate the cause of the watchdog entries.'),
            ];
          }
          else {
            $validation[$key] = [
              'class' => 'ok',
              'text' => $this->t('OK'),
            ];
          }

          break;

        case 'sgd_watchdog_total_debug':

          if ($status['value'] > 0) {
            $validation[$key] = [
              'class' => 'warning',
              'text' => $this->t('Alert'),
              'message' => $this->t('Debug messages should not be active on production environments.'),
            ];
          }
          else {
            $validation[$key] = [
              'class' => 'ok',
              'text' => $this->t('OK'),
            ];
          }

          break;

      }

    }

    return $validation;
  }

}
