<?php

namespace Drupal\sgd_dashboard\Entity\Bundle;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\sgd_dashboard\Entity\SgdWebsiteData;
use Drupal\user\Entity\User;

/**
 * A bundle class for node entities.
 */
class WebsiteBundle extends NodeBundle {

  use StringTranslationTrait;

  /**
   * Returns the title for the website node.
   *
   * This is a concatonation of the node name, the environment and the suffix
   * if there is one.
   */
  public function getTitle(): TranslatableMarkup {

    $title = trim($this->get('title')->value);

    if (!empty($terms = $this->get('field_environment_type')->referencedEntities())) {
      $environmentType = $terms[0]->getName();
    }
    else {
      $environmentType = '';
    }

    $suffix = isset($this->get('field_title_suffix')->value) ? '(' . trim($this->get('field_title_suffix')->value, " ()\t\n\r\0\x0B") . ')' : '';

    return $this->t('@title (@env) @suffix', [
      '@title' => $title,
      '@env' => $environmentType,
      '@suffix' => $suffix,
    ]);

  }

  /**
   * Returns the website title.
   *
   * This is in a sanitized form suitable for use as a filename.
   *
   * @return string
   *   The sanitized title.
   */
  public function getFileNameFromTitle(): string {

    // Get and sanitise the website title.
    $special_chars = [
      "?", "[", "]", "/", "\\", "=", "<", ">", ":", ";", ",", "'", "\"", "&",
      "$", "#", "*", "(", ")", "|", "~", "`", "!", "{", "}", "%", "+", chr(0),
    ];

    $filename = trim($this->getTitle());
    $filename = preg_replace("#\x{00a0}#siu", ' ', $filename);
    $filename = str_replace($special_chars, '', $filename);
    $filename = str_replace(['%20', '+'], '-', $filename);
    $filename = preg_replace('/[\r\n\t -]+/', '-', $filename);
    $filename = trim($filename, '.-_');
    $filename = strtolower($filename);

    return $filename;
  }

  /**
   * Builds the URL for the Site Guardian API.
   *
   * @return string
   *   The URL built from the website node.
   */
  public function getBaseUrl() : string {

    // Get the values we need from the node and tidy them up.
    $baseUrl = !empty($this->get('field_url')->value) ? trim($this->get('field_url')->value, " \t/") : '';

    // No URL then return.
    if (empty($baseUrl)) {
      return FALSE;
    }

    return $baseUrl;
  }

  /**
   * Returns basic auth username and password if it exists.
   *
   * @return array
   *   The basic auth username and password or false.
   */
  public function getBasicAuth() : FALSE | array {

    $user = User::load(\Drupal::currentUser()->id());

    // If user has appropriate permission then add any basic auth.
    if ($this->access('update', $user)) {
      $username = !empty($this->get('field_http_auth_user')->value) ? trim($this->get('field_http_auth_user')->value, " \t") : '';
      $password = !empty($this->get('field_http_auth_password')->value) ? trim($this->get('field_http_auth_password')->value, " \t") : '';
    }

    // If no user name then return false
    if (empty($username)) {
      return FALSE;
    }

    return [
      'username' => $username,
      'password' => $password
    ];
  }

  /**
   * Gets the website data storage entity for the website.
   *
   * @return Drupal\sgd_dashboard\Entity\SgdWebsiteData
   *   A Site Guardian Website data entity.
   */
  public function getDataEntity(): SgdWebsiteData {
    return $this->field_sgd_website_data->entity;
  }

  /**
   * Adds Basic Authentication credentials to a URL.
   *
   * @return string
   *   Passed URL with username and password inserted.
   */
  private function addBasicAuthToUrl($url, $username, $password) : string {

    if (empty($username) || empty($password)) {
      return $url;
    }

    $parsedUrl = parse_url($url);

    if (isset($parsedUrl['scheme']) && isset($parsedUrl['host'])) {

      $url = $parsedUrl['scheme'] . '://' . $username . ':' . $password . '@' . $parsedUrl['host'];

      if (isset($parsedUrl['port'])) {
        $url .= ':' . $parsedUrl['port'];
      }

      if (isset($parsedUrl['path'])) {
        $url .= $parsedUrl['path'];
      }

      if (isset($parsedUrl['query'])) {
        $url .= '?' . $parsedUrl['query'];
      }

      if (isset($parsedUrl['fragment'])) {
        $url .= '#' . $parsedUrl['fragment'];
      }
    }

    return $url;
  }

}
