<?php

namespace Drupal\sgd_pdf_report\Plugin\QueueWorker;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Mail\MailManager;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\sgd_dashboard\Entity\Bundle\WebsiteBundle;
use Drupal\sgd_pdf_report\Services\WebsitePdfGeneratorService;
use Drupal\sgd_pdf_report\Services\WebsiteReportService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Queue worker for Site Guardian Dashboard websites.
 *
 * @QueueWorker(
 *   id = "site_guardian_send_report",
 *   title = @Translation("Site Guardian PDF Report Queue Worker"),
 *   cron = {"time" = 60}
 * )
 */
class SgdSendReportQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * The node storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $nodeStorage;

  /**
   * The mail manager.
   *
   * @var \Drupal\Core\Mail\MailManager
   */
  protected $mailManager;

  /**
   * The Website report service.
   *
   * @var \Drupal\sgd_pdf_report\Services\WebsiteReportService
   */
  protected $websiteReport;

  /**
   * The PDF generator service.
   *
   * @var \Drupal\sgd_pdf_report\Services\WebsitePdfGeneratorService
   */
  protected $pdfGenerator;

  /**
   * Logging channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    LoggerChannelInterface $logger,
    EntityStorageInterface $node_storage,
    MailManager $mail_manager,
    WebsitePdfGeneratorService $pdf_generator,
    WebsiteReportService $website_report,
  ) {

    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->logger = $logger;
    $this->nodeStorage = $node_storage;
    $this->mailManager = $mail_manager;
    $this->pdfGenerator = $pdf_generator;
    $this->websiteReport = $website_report;
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('logger.factory')->get('logger.channel.sgd_pdf_report'),
      $container->get('entity_type.manager')->getStorage('node'),
      $container->get('plugin.manager.mail'),
      $container->get('sgd_pdf_report.website_pdf_generator'),
      $container->get('sgd_pdf_report.website_report')
    );
  }

  /**
   * {@inheritDoc}
   */
  public function processItem($data) {

    /** @var \Drupal\sgd_dashboard\Entity\Bundle\WebsiteBundle $websiteNode */
    $websiteNode = $this->nodeStorage->load($data['webSiteNodeId']);

    // If the site is still published.
    if ($websiteNode->isPublished() && $websiteNode instanceof WebsiteBundle) {

      $webSiteTitle = $websiteNode->getTitle();

      // Get the recipients.
      $recipients = $websiteNode->getReportRecipientEmails();

      $this->logger->notice(
        'Emailing Website PDF report for @website to @emails',
        [
          '@website' => $webSiteTitle,
          '@emails' => $recipients,
        ],
      );

      // We have the website so generate the report.
      // Get current date time.
      $now = \Drupal::service('date.formatter')->format(\Drupal::time()->getCurrentTime(), 'custom', 'Ymd-His');

      // Get the sanitized form of the websites title.
      $sanitizedFileName = $websiteNode->getFileNameFromTitle();

      $fileName = $now . "-" . $sanitizedFileName;

      $reportTitle = "Site Guardian Website Report for " . $webSiteTitle;

      $build = $this->websiteReport->getBuildArray($websiteNode);

      $cssPath = \Drupal::service('extension.list.module')->getPath('sgd_pdf_report') . '/css/pdf-report.css';

      $savedFilePath = $this->pdfGenerator->saveTofile($fileName, $reportTitle, $build, cssPath: $cssPath);

      // Setup the Email.
      $params['title'] = $this->t("Site Guardian Website report for @website", [
        '@website' => $webSiteTitle,
      ]);

      $params['message'] = $this->t("Please find attached the Site Guardian Website report for @website", [
        '@website' => $webSiteTitle,
      ]);

      // Here we add our files URIs for an attachment.
      $params['attachments'] = [
        $savedFilePath,
      ];

      $langcode = \Drupal::currentUser()->getPreferredLangcode();

      // Email the report to the receipients.
      $this->mailManager->mail('sgd_pdf_report', 'website_pdf_report', $recipients, $langcode, $params, NULL, TRUE);

      // Delete the temporary file.
      unlink($savedFilePath);
    }
  }

}
