<?php

namespace Drupal\sgd_pdf_report\Controller;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Url;
use Drupal\sgd_dashboard\Entity\Bundle\WebsiteBundle;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Provides route responses.
 */
class SgdPDFReportController extends ControllerBase {

  /**
   * The Website report service.
   *
   * @var \Drupal\sgd_pdf_report\Services\websiteReportService
   */
  protected $websiteReport;

  /**
   * The PDF generator service.
   *
   * @var \Drupal\sgd_pdf_report\Services\WebsitePdfGeneratorService
   */
  protected $pdfGenerator;

  /**
   * The date formatter interface.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The module extension list.
   */
  protected ModuleExtensionList $moduleExtensionList;

  /**
   * The system time service.
   */
  protected TimeInterface $time;

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container) {

    $instance = parent::create($container);

    $instance->pdfGenerator = $container->get('sgd_pdf_report.website_pdf_generator');
    $instance->websiteReport = $container->get('sgd_pdf_report.website_report');
    $instance->dateFormatter = $container->get('date.formatter');
    $instance->moduleExtensionList = $container->get('extension.list.module');
    $instance->time = $container->get('datetime.time');

    return $instance;
  }

  /**
   * Build a website PDF report.
   *
   * Generates a website PDF report and downloads it to the users browser.
   */
  public function downloadReport(WebsiteBundle $node, RouteMatchInterface $route_match) {

    // Make sure the site has been scanned as not much point generating a
    // report if it hasn't.
    if (!empty($node->field_site_last_checked->value)) {

      $cssPath = $this->moduleExtensionList->getPath('sgd_pdf_report') . '/css/pdf-report.css';

      // We have the website so generate the report.
      $build = $this->websiteReport->getBuildArray($node);

      $reportTitle = "Site Guardian Website Report for " . $node->getTitle();

      // Get current date time.
      $now = $this->dateFormatter->format($this->time->getCurrentTime(), 'custom', 'Ymd-His');

      // Get the sanitized form of the websites title.
      $sanitizedFileName = $node->getFileNameFromTitle();

      $fileName = $now . "-" . $sanitizedFileName;

      $response = $this->pdfGenerator->getResponse($fileName, $reportTitle, $build, cssPath: $cssPath);

      return $response;
    }

    // If the site has not been scanned and we somehow got here then redirect
    // to the websites canonical route. A bit nicer than an error or denying
    // access.
    else {

      $url = Url::fromRoute('entity.node.canonical', ['node' => $node->id()])->toString();

      return new RedirectResponse($url);
    }

  }

}
