<?php

namespace Drupal\sgd_pdf_report\Services;

use Dompdf\Dompdf;
use Dompdf\Options;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\File\FileExists;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Render\Renderer;
use Drupal\Core\Render\Markup;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\File\FileRepositoryInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;

/**
 * Defines the WebsitePdfGeneratorService service.
 */
class WebsitePdfGeneratorService {

  use StringTranslationTrait;

  /**
   * Options for dompdf.
   *
   * @var \Dompdf\Options
   */
  protected $options;

  /**
   * Dompdf library.
   *
   * @var \Dompdf\Dompdf
   */
  protected $dompdf;

  /**
   * The object renderer.
   *
   * @var \Drupal\Core\Render\Renderer
   */
  protected $renderer;

  /**
   * Protected requestStack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The module extension list.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected $moduleExtensionList;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The file repository service.
   *
   * @var \Drupal\file\FileRepositoryInterface
   */
  protected $fileRepository;

  /**
   * Constructs a new object.
   *
   * @param \Drupal\Core\Render\Renderer $renderer
   *   The object renderer.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The current request stack.
   * @param \Drupal\Core\Extension\ModuleExtensionList $extension_list_module
   *   The module extension list.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\file\FileRepositoryInterface $file_repository
   *   The file repository service.
   */
  public function __construct(
    Renderer $renderer,
    RequestStack $request_stack,
    ModuleExtensionList $extension_list_module,
    FileSystemInterface $file_system,
    FileRepositoryInterface $file_repository,
  ) {

    $this->renderer = $renderer;
    $this->requestStack = $request_stack;
    $this->moduleExtensionList = $extension_list_module;
    $this->fileSystem = $file_system;
    $this->fileRepository = $file_repository;

    $this->options = new Options();

    $this->options->set('enable_css_float', TRUE);
    $this->options->set('enable_html5_parser', TRUE);
    $this->options->set('enable_remote', TRUE);
    $this->options->set('defaultFont', 'Times');
    $this->options->set('chroot', DRUPAL_ROOT);
  }

  /**
   * Save to file using dompdf with given options.
   *
   * @param string $fileName
   *   The filename to use when saving the PDF.
   * @param string $title
   *   The title of the pdf.
   * @param array $content
   *   The build array to be rendered.
   * @param bool $preview
   *   If enabled the html will be printed without render the pdf.
   * @param array $options
   *   The options array to perform pdf.
   * @param string $pageSize
   *   The size of the page.
   * @param string $showPagination
   *   Whether or not to show pagination.
   * @param string $paginationX
   *   X Location for pagination.
   * @param string $paginationY
   *   Y Location for pagination.
   * @param string $disposition
   *   The disposition of the page, portrait or landscape.
   * @param string $cssText
   *   Text to load additional css.
   * @param string $cssPath
   *   Path to load additional css.
   */
  public function saveTofile($fileName, $title, array $content, $preview = FALSE, array $options = [], $pageSize = 'A4', $showPagination = 0, $paginationX = 0, $paginationY = 0, $disposition = 'portrait', $cssText = NULL, $cssPath = NULL) {

    $output = $this->getOutput($title, $content, $preview, $options, $pageSize, $showPagination, $paginationX, $paginationY, $disposition, $cssText, $cssPath, FALSE);

    $folder = "temporary://pdf-reports";

    $this->fileSystem->prepareDirectory($folder, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);

    $filePath = $folder . "/" . $fileName . ".pdf";

    // If the FileExists::class enum is available use it
    // (available from 10.3 onwards).
    if (enum_exists(FileExists::class)) {
      $this->fileRepository->writeData($output, $filePath, FileExists::Replace);
    }
    // Otherwise use FileSystemInterface::EXISTS_REPLACE
    // Deprecated in 10.3 will be removed in 12.0.
    else {
      // @phpstan-ignore-next-line
      $this->fileRepository->writeData($output, $filePath, FileSystemInterface::EXISTS_REPLACE);
    }

    return $filePath;
  }

  /**
   * Render using dompdf with given options.
   *
   * @param string $fileName
   *   The filename to use when downloading the PDF.
   * @param string $title
   *   The title of the pdf.
   * @param array $content
   *   The build array to be rendered.
   * @param bool $preview
   *   If enabled the html will be printed without render the pdf.
   * @param array $options
   *   The options array to perform pdf.
   * @param string $pageSize
   *   The size of the page.
   * @param string $showPagination
   *   Whether or not to show pagination.
   * @param string $paginationX
   *   X Location for pagination.
   * @param string $paginationY
   *   Y Location for pagination.
   * @param string $disposition
   *   The disposition of the page, portrait or landscape.
   * @param string $cssText
   *   Text to load additional css.
   * @param string $cssPath
   *   Path to load additional css.
   * @param bool $forceDownload
   *   If force download by Content-Disposition header is required.
   */
  public function getResponse($fileName, $title, array $content, $preview = FALSE, array $options = [], $pageSize = 'A4', $showPagination = 0, $paginationX = 0, $paginationY = 0, $disposition = 'portrait', $cssText = NULL, $cssPath = NULL, $forceDownload = TRUE) {

    $output = $this->getOutput($title, $content, $preview, $options, $pageSize, $showPagination, $paginationX, $paginationY, $disposition, $cssText, $cssPath, $forceDownload);

    $response = new Response();

    $response->setContent($output);

    $response->headers->set('Content-Type', 'application/pdf');

    if (is_array($title)) {
      $title = $this->renderer->render($title);
    }

    if ($forceDownload) {
      $response->headers->set('Content-Disposition', "attachment; filename={$fileName}.pdf");
    }

    return $response;
  }

  /**
   * Render usin dompdf with given options.
   *
   * @param string $title
   *   The title of the pdf.
   * @param array $content
   *   The build array to be rendered.
   * @param bool $preview
   *   If enabled the html will be printed without render the pdf.
   * @param array $options
   *   The options array to perform pdf.
   * @param string $pageSize
   *   The size of the page.
   * @param string $showPagination
   *   Whether or not to show pagination.
   * @param string $paginationX
   *   X Location for pagination.
   * @param string $paginationY
   *   Y Location for pagination.
   * @param string $disposition
   *   The disposition of the page, portrait or landscape.
   * @param string $cssText
   *   Text to load additional css.
   * @param string $cssPath
   *   Path to load additional css.
   * @param bool $forceDownload
   *   If force download by Content-Disposition header is required.
   */
  private function getOutput($title, array $content, $preview = FALSE, array $options = [], $pageSize = 'A4', $showPagination = 0, $paginationX = 0, $paginationY = 0, $disposition = 'portrait', $cssText = NULL, $cssPath = NULL, $forceDownload = TRUE) {

    $request = $this->requestStack->getCurrentRequest();
    $base_url = $request->getSchemeAndHttpHost();

    // By default we load some options.
    // The user can override these options.
    foreach ($options as $key => $option) {
      $this->options->set($key, $option);
    }

    // Correct temporary and folders to write fonts because the vendor folder
    // that is the default normally can't be writtable.
    $fontDir = 'public://dompdf-fonts';
    $cacheDir = 'public://dompdf-fonts/cache';

    $this->fileSystem->prepareDirectory($fontDir, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);
    $this->fileSystem->prepareDirectory($cacheDir, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);

    $fontDir = $this->fileSystem->realPath($fontDir);
    $cacheDir = $this->fileSystem->realPath($cacheDir);
    $tmpDir = $this->fileSystem->realPath('temporary://');

    $this->options->set('fontDir', $fontDir);
    $this->options->set('fontCache', $cacheDir);
    $this->options->set('tempDir', $tmpDir);

    // Dompdf needs to be initialized with custom options if they are supplied.
    $this->dompdf = new Dompdf($this->options);
    $path = $this->moduleExtensionList->getPath('sgd_pdf_report');
    $css = file_get_contents($path . '/css/pdf.css');

    // Add inline css from text.
    if (!empty($cssText)) {
      $css .= "\r\n";
      $css .= $cssText;
    }
    // Add inline css from file.
    if (!empty($cssPath) && file_exists($cssPath)) {
      $css .= "\r\n";
      $css .= file_get_contents($cssPath);
    }

    $css = str_replace("url('/", "url('" . $base_url . "/", $css);

    $build = [
      '#theme' => 'pdf_generator_print',
      '#css' => Markup::create($css),
      '#content' => $content,
      '#title' => $title,
    ];

    if ($preview) {
      return $build;
    }
    $html = (string) $this->renderer->renderRoot($build);

    $html = str_replace('src="' . $base_url . '/', 'src="/', $html);
    $html = str_replace('href="/', 'href="' . $base_url . '/', $html);
    $html = str_replace('src="/', 'src="' . DRUPAL_ROOT . '/', $html);

    $this->dompdf->setOptions($this->options);
    $this->dompdf->loadHtml($html);
    $this->dompdf->setPaper($pageSize, $disposition);

    $this->dompdf->render();

    switch ($showPagination) {

      case 1:
        $canvas = $this->dompdf->getCanvas();
        $canvas->page_text($paginationX, $paginationY, "{PAGE_NUM}", NULL, 12);
        $this->dompdf->setCanvas($canvas);
        break;

      case 2:
        $canvas = $this->dompdf->getCanvas();
        $canvas->page_text($paginationX, $paginationY, "{PAGE_NUM}/{PAGE_COUNT}", NULL, 12);
        $this->dompdf->setCanvas($canvas);
        break;
    }

    return $this->dompdf->output();
  }

}
