<?php

namespace Drupal\sgd_pdf_report\Services;

use Drupal\Component\Datetime\Time;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\sgd_dashboard\Entity\Bundle\WebsiteBundle;
use Drupal\sgd_dashboard\SgdCompanionPluginManager;

/**
 * Defines the Website report service.
 */
class WebsiteReportService {

  use StringTranslationTrait;

  /**
   * The SGD Companion plugin manager.
   *
   * @var \Drupal\sgd_dashboard\SgdCompanionPluginManager
   */
  protected $pluginManager;

  /**
   * The date formatter interface.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The system time service.
   */
  protected TimeInterface $time;

  public function __construct(
    SgdCompanionPluginManager $pluginManager,
    DateFormatterInterface $dateFormatter,
    Time $time,
  ) {
    $this->pluginManager = $pluginManager;
    $this->dateFormatter = $dateFormatter;
    $this->time = $time;
  }

  /**
   * Defines the WebsiteReportService service.
   */
  public function getBuildArray(WebsiteBundle $website) {

    /** @var \Drupal\sgd_dashboard\Entity\SgdWebsiteData $websiteData */
    $websiteData = $website->getDataEntity();

    // Find all Companion plugins and iterate through them calling their
    // buildElements function.
    // Get all the companion modules.
    $plugins = $this->pluginManager->getDefinitions();

    // Iterate over all instances and get the elements we can use in the
    // PDF report pages.
    foreach ($plugins as $plugin) {

      $instance = $this->pluginManager->createInstance($plugin['id']);

      if ($elements = $instance->getBuildElements($websiteData)) {
        $pluginElements[$plugin['id']] = $elements;
      }

    }

    // Now build the report.
    // First the general website stuff.
    $build['website_data'] = [
      '#theme' => 'sgd_website_report',
      '#data' => [
        'title' => trim($website->getTitle()),
        'report_date' => $this->dateFormatter->format($this->time->getCurrentTime(), 'custom', 'l jS \of F Y \a\t H:i:s'),
        'client' => $website->field_client->entity->title->value,
        'environment_type' => $website->field_environment_type->entity->name->value ?? 'Not defined',
        'url' => $website->getBaseUrl(FALSE),
      ],
    ];

    // Make the scan date sensible, formatted if there is a value or null if
    // it's never been scanned.
    if (!empty($website->field_site_last_checked->value)) {
      $build['website_data']['#data']['scan_date'] = $this->dateFormatter->format($website->field_site_last_checked->value, 'custom', 'l jS \of F Y \a\t H:i:s');
    }
    else {
      $build['website_data']['#data']['scan_date'] = NULL;
    }

    // Add the core status info.
    $build['website_data']['#data'] += $pluginElements['sgd_companion_sgd_core'];

    // Now project data.
    if (!empty($pluginElements['sgd_companion_sgd_projects'])) {

      $build['project_data'] = [
        '#theme' => 'sgd_projects_report',
        '#data' => $pluginElements['sgd_companion_sgd_projects'],
      ];

      // Special case - Uninstalled project is not in the project data because thats
      // only enabled projects but it makes more sense to show it in the project info in
      // the report.
      if (!empty($pluginElements['sgd_companion_sgd_core']['projects']['uninstalled'])) {
        $build['project_data']['#data']['sgd_projects_uninstalled'] = $pluginElements['sgd_companion_sgd_core']['projects']['uninstalled'];
      }
    }

    // Now add user/role status.
    if (!empty($pluginElements['sgd_companion_user_status'])) {
      $build['user_data'] = [
        '#theme' => 'sgd_users_report',
        '#data' => $pluginElements['sgd_companion_user_status'],
      ];
    }

    // Now PHP status and memory usage.
    if (!empty($pluginElements['sgd_companion_php_status']) || !empty($pluginElements['sgd_companion_memory_profiler_plus'])) {

      $build['php_data'] = [
        '#theme' => 'sgd_php_report',
      ];

      if (!empty($pluginElements['sgd_companion_php_status'])) {
        $build['php_data']['#data']['status'] = $pluginElements['sgd_companion_php_status'];
      }

      if (!empty($pluginElements['sgd_companion_memory_profiler_plus'])) {
        $build['php_data']['#data']['memory_usage'] = $pluginElements['sgd_companion_memory_profiler_plus'];
      }
    }

    // Now watchdog summary.
    if (!empty($pluginElements['sgd_companion_watchdog_summary'])) {
      $build['watchdog_summary'] = [
        '#theme' => 'sgd_watchdog_summary_report',
        '#data' => $pluginElements['sgd_companion_watchdog_summary'],
      ];
    }

    // Now Benchmark data.
    if (!empty($pluginElements['sgd_companion_server_benchmarks'])) {
      $build['benchmark_data'] = [
        '#theme' => 'sgd_benchmark_report',
        '#data' => $pluginElements['sgd_companion_server_benchmarks'],
      ];
    }

    return $build;
  }

}
