<?php

namespace Drupal\sgd_pdf_report\Plugin\QueueWorker;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\sgd_dashboard\Entity\Bundle\WebsiteBundle;
use Drupal\sgd_pdf_report\Services\WebsitePdfGeneratorService;
use Drupal\sgd_pdf_report\Services\WebsiteReportService;
use Drupal\sgd_pdf_report\Services\ReportDistributionService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Queue worker for Site Guardian Dashboard websites.
 *
 * @QueueWorker(
 *   id = "site_guardian_send_report",
 *   title = @Translation("Site Guardian PDF Report Queue Worker"),
 *   cron = {"time" = 60}
 * )
 */
class SgdSendReportQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * The node storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $nodeStorage;

  /**
   * The Website report service.
   *
   * @var \Drupal\sgd_pdf_report\Services\WebsiteReportService
   */
  protected $websiteReport;

  /**
   * The PDF generator service.
   *
   * @var \Drupal\sgd_pdf_report\Services\WebsitePdfGeneratorService
   */
  protected $pdfGenerator;

  /**
   * The Report Distribution service.
   * This may be overridden if more sophisticated mechanisms for determining who gets a 
   * report or how is required.
   *
   * @var \Drupal\sgd_pdf_report\Services\ReportDistributionService
   */
  protected $reportDistributionService;

  /**
   * Logging channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    LoggerChannelInterface $logger,
    EntityStorageInterface $node_storage,
    WebsitePdfGeneratorService $pdf_generator,
    WebsiteReportService $website_report,
    ReportDistributionService $report_distribution_service,
  ) {

    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->logger = $logger;
    $this->nodeStorage = $node_storage;
    $this->pdfGenerator = $pdf_generator;
    $this->websiteReport = $website_report;
    $this->reportDistributionService = $report_distribution_service;
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('logger.factory')->get('logger.channel.sgd_pdf_report'),
      $container->get('entity_type.manager')->getStorage('node'),
      $container->get('sgd_pdf_report.website_pdf_generator'),
      $container->get('sgd_pdf_report.website_report'),
      $container->get('sgd_pdf_report.report_distribution_service'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function processItem($data) {

    /** @var \Drupal\sgd_dashboard\Entity\Bundle\WebsiteBundle $websiteNode */
    $websiteNode = $this->nodeStorage->load($data['webSiteNodeId']);

    // If the site is still published.
    if ($websiteNode->isPublished() && $websiteNode instanceof WebsiteBundle) {

      $webSiteTitle = $websiteNode->getTitle();

      // Get the recipients and if there are any build the report.
      if ($userEmails = $this->reportDistributionService->getReportRecipients($websiteNode)) {

        $recipients = implode(',', $userEmails);

        $this->logger->notice(
          'Emailing Website PDF report for @website to @emails',
          [
            '@website' => $webSiteTitle,
            '@emails' => $recipients,
          ],
        );

        // We have the website so generate the report.
        // Get current date time.
        $now = \Drupal::service('date.formatter')->format(\Drupal::time()->getCurrentTime(), 'custom', 'Ymd-His');

        // Get the sanitized form of the websites title.
        $sanitizedFileName = $websiteNode->getFileNameFromTitle();

        $fileName = $now . "-" . $sanitizedFileName;

        $reportTitle = "Site Guardian Website Report for " . $webSiteTitle;

        $build = $this->websiteReport->getBuildArray($websiteNode);

        $cssPath = \Drupal::service('extension.list.module')->getPath('sgd_pdf_report') . '/css/pdf-report.css';

        $savedFilePath = $this->pdfGenerator->saveTofile($fileName, $reportTitle, $build, cssPath: $cssPath);

        // Now actually distribute the report.
        $this->reportDistributionService->sendReport($websiteNode, $savedFilePath);

        // Delete the temporary file.
        unlink($savedFilePath);
      }
    }
  }

}
