<?php

namespace Drupal\sgd_pdf_report\Services;

use Drupal\Component\Datetime\Time;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Mail\MailManager;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\sgd_dashboard\Entity\Bundle\WebsiteBundle;

/**
 * Defines the Report distribution service.
 */
class ReportDistributionService {

  use StringTranslationTrait;

  /**
   * The mail manager.
   *
   * @var \Drupal\Core\Mail\MailManager
   */
  protected $mailManager;

  /**
   * The date formatter interface.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The system time service.
   */
  protected TimeInterface $time;

  public function __construct(MailManager $mail_manager, DateFormatterInterface $dateFormatter, Time $time) {
    $this->mailManager = $mail_manager;
    $this->dateFormatter = $dateFormatter;
    $this->time = $time;
  }

  /**
   * Indicates whether the report can be distrubuted for the passed
   * website.
   * This can be used to stop queueing of jobs for any arbitary reason
   * you may require.
   *
   * @return boolean
   *   Whether distrubution should occur
   */
  public function canDistributeReport(WebsiteBundle $website): bool {
    return TRUE;
  }

  /**
   * Gets any report recipients that have been specified for the website.
   *
   * Returned as a comma seperated string.
   *
   * @return array
   *   An array of email addresses.
   */
  public function getReportRecipients(WebsiteBundle $website): array {

    $userEmails = [];

    foreach ($website->field_report_recipients as $user) {
      $userEmails[] = $user->entity->mail->value;
    }

    return $userEmails;
  }

  /**
   * Sends the report to recipients that have been specified for the website.
   */
  public function sendReport(WebsiteBundle $websiteNode, string $savedFilePath): void {

    $webSiteTitle = $websiteNode->getTitle();

    // Setup the Email.
    $params['title'] = $this->t("Site Guardian Website report for @website", [
      '@website' => $webSiteTitle,
    ]);

    $params['message'] = $this->t("Please find attached the Site Guardian Website report for @website", [
      '@website' => $webSiteTitle,
    ]);

    // Here we add our files URIs for an attachment.
    $params['attachments'][] = [
      'filepath' => $savedFilePath,
      'filename' => \Drupal::service('file_system')->basename($savedFilePath),
      'filemime' => "application/pdf",
    ];

    $replyTo = \Drupal::config('system.site')->get('mail');

    $langcode = \Drupal::currentUser()->getPreferredLangcode();

    // Email the report to the recipients.
    $userEmails = $this->getReportRecipients($websiteNode);

    $recipients = implode(',', $userEmails);

    $this->mailManager->mail('sgd_pdf_report', 'website_pdf_report', $recipients, $langcode, $params, $replyTo, TRUE);
  }

}
