<?php

namespace Drupal\shareable_sdc;

use Drupal\Component\Discovery\YamlDirectoryDiscovery;
use Drupal\Core\Plugin\Discovery\DirectoryWithMetadataPluginDiscovery;
use Drupal\Core\Theme\ComponentPluginManager;

/**
 * Extended component plugin manager that supports global /components directory.
 *
 * This manager extends the core ComponentPluginManager to add support for
 * components located in a global /components directory, accessible via
 * the "components:" namespace in Twig templates.
 */
class ShareableComponentPluginManager extends ComponentPluginManager {

  /**
   * {@inheritdoc}
   */
  protected function getDiscovery(): DirectoryWithMetadataPluginDiscovery {
    // Basically this is an exact copy of the parent method.
    if (!isset($this->discovery)) {
      $directories = $this->getScanDirectories();
      $this->discovery = new DirectoryWithMetadataPluginDiscovery($directories, 'component', $this->fileSystem);
    }
    return $this->discovery;
  }

  /**
   * Get the list of directories to scan.
   *
   * Because the parent class has this method as private instead of protected, we cannot override it. So we have to
   * create our own version (which is almost a copy of the parent class) with the extension for our custom code.
   *
   * @return string[]
   *   The directories.
   */
  protected function getScanDirectories(): array {
    // Get the original extension directories (modules and themes)
    $extension_directories = [
      ...$this->moduleHandler->getModuleDirectories(),
      ...$this->themeHandler->getThemeDirectories(),
    ];

    $directories = array_map(
      static fn(string $path) => rtrim($path, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . 'components',
      $extension_directories
    ) + ['components' => rtrim($this->appRoot, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . 'components'];

    return $directories;

  }

  /**
   * {@inheritdoc}
   */
  protected function alterDefinition(array $definition): array {
    $metadata_path = $definition[YamlDirectoryDiscovery::FILE_KEY] ?? '';

    // Let the parent implementation handle it if not one of the global components.
    if (!str_contains($metadata_path, $this->appRoot . DIRECTORY_SEPARATOR . 'components')) {
      return parent::alterDefinition($definition);
    }

    // Then sets the definition for our custom type.
    $definition['provider'] = 'components';
    $definition['extension_type'] = 'theme';

    // Override the plugin ID to use the "components:" namespace
    [, $machine_name] = explode(':', $definition['id']);
    $definition['id'] = 'components:' . $machine_name;

    // Handle global component definition early to avoid theme checks.
    $component_directory = $this->fileSystem->dirname($metadata_path);
    $definition['path'] = $component_directory;
    $definition['machineName'] = $machine_name;
    $definition['library'] = $this->libraryFromDefinition($definition);

    // Discover the template
    $template = $this->findAsset(
      $component_directory,
      $definition['machineName'],
      'twig'
    );
    $definition['template'] = basename($template);
    $definition['documentation'] = 'No documentation found. Add a README.md in your component directory.';
    $documentation_path = sprintf('%s/README.md', $this->fileSystem->dirname($metadata_path));
    if (file_exists($documentation_path)) {
      $definition['documentation'] = file_get_contents($documentation_path);
    }

    return $definition;
  }

  /**
   * {@inheritdoc}
   */
  protected function providerExists($provider) {
    return parent::providerExists($provider) || $provider == 'components';
  }

  /**
   * Finds assets related to the provided metadata file.
   *
   * @see Drupal\Core\Theme\ComponentPluginManager::findAsset().
   */
  private function findAsset(string $component_directory, string $machine_name, string $file_extension, bool $make_relative = FALSE): ?string {
    $absolute_path = sprintf('%s%s%s.%s', $component_directory, DIRECTORY_SEPARATOR, $machine_name, $file_extension);
    if (!file_exists($absolute_path)) {
      return NULL;
    }
    return $make_relative
      ? $this->makePathRelativeToLibraryRoot($absolute_path)
      : $absolute_path;
  }

  /**
   * Takes a path and makes it relative to the library provider.
   *
   * @see Drupal\Core\Theme\ComponentPluginManager::makePathRelativeToLibraryRoot().
   */
  private function makePathRelativeToLibraryRoot(string $path): string {
    $path_from_root = str_starts_with($path, $this->appRoot)
      ? substr($path, strlen($this->appRoot) + 1)
      : $path;
    // Make sure this works seamlessly in every OS.
    $path_from_root = str_replace(DIRECTORY_SEPARATOR, '/', $path_from_root);
    // The library owner is in <root>/core, so we need to go one level up to
    // find the app root.
    return '../' . $path_from_root;
  }
}
