<?php

namespace Drupal\sharepoint_integration\API;

use Drupal\Core\Config\ConfigFactoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;

/**
 * Service for all interactions with the Microsoft Graph API.
 */
class GraphApiService {

  /**
   * The base URL for the Microsoft Graph API.
   */
  const GRAPH_API_BASE_URL = 'https://graph.microsoft.com/v1.0';

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The Token Fetcher service.
   *
   * @var \Drupal\sharepoint_integration\API\TokenFetcher
   */
  protected $tokenFetcher;

  /**
   * The exception logger.
   *
   * @var \Drupal\sharepoint_integration\API\ExceptionLogger
   */
  protected $exceptionLogger;

  /**
   * Constructs a new GraphApiService object.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The Guzzle HTTP client.
   * @param \Drupal\sharepoint_integration\API\TokenFetcher $token_fetcher
   *   The service to fetch the access token.
   * @param \Drupal\sharepoint_integration\API\ExceptionLogger $exception_logger
   *   The exception logger service.
   */
  public function __construct(ClientInterface $http_client, TokenFetcher $token_fetcher, ExceptionLogger $exception_logger) {
    $this->httpClient = $http_client;
    $this->tokenFetcher = $token_fetcher;
    $this->exceptionLogger = $exception_logger;
  }

  /**
   * Fetches all SharePoint sites the configured app can access.
   *
   * @return array
   *   An array of SharePoint sites.
   */
  public function getSites(): array {
    // Using $select for better performance.
    $endpoint = '/sites?search=*&$select=id,displayName,webUrl';
    return $this->makeRequest('GET', $endpoint);
  }

  /**
   * Fetches all document libraries (drives) for a given site.
   *
   * @param string $siteId
   *   The ID of the SharePoint site.
   *
   * @return array
   *   An array of document libraries.
   */
  public function getDrives(string $siteId): array {
    // Using $select for better performance.
    $endpoint = "/sites/{$siteId}/drives?\$select=id,name,webUrl,driveType,sharepointIds";
    return $this->makeRequest('GET', $endpoint);
  }

  /**
   * Fetches the contents of a drive or a specific folder within a drive.
   *
   * @param string $driveId
   *   The ID of the document library (drive).
   * @param string $folderId
   *   (Optional) The ID of the folder. Defaults to 'root'.
   * @param array $customColumns
   *   (Optional) An array of custom column names to retrieve.
   *
   * @return array
   *   An array of drive items (files and folders).
   */
  public function getDriveItems(string $driveId, string $folderId = 'root', array $customColumns = []): array {
    if ($folderId === 'root') {
      $endpoint = "/drives/{$driveId}/root/children";
    }
    else {
      $endpoint = "/drives/{$driveId}/items/{$folderId}/children";
    }
    $options = [];

    // If custom columns are requested, expand the listItem and its fields. [1, 2]
    if (!empty($customColumns)) {
      $selectFields = implode(',', $customColumns);
      $options['query'] = [
        '$expand' => "listItem(expand=fields(select={$selectFields}))",
      ];
    }

    return $this->makeRequest('GET', $endpoint, $options);
  }

  /**
   * Fetches the available columns for a document library (list).
   *
   * @param string $siteId
   *   The ID of the site.
   * @param string $listId
   *   The ID of the list (can be retrieved from a drive's sharepointIds).
   *
   * @return array
   *   An array of available column definitions.
   */
  public function getAvailableColumns(string $siteId, string $listId): array {
    $endpoint = "/sites/{$siteId}/lists/{$listId}/columns";
    return $this->makeRequest('GET', $endpoint);
  }

  /**
   * Gets the embeddable preview URL for a file.
   *
   * @param string $driveId
   *   The ID of the drive containing the item.
   * @param string $itemId
   *   The ID of the file item.
   *
   * @return array
   *   The preview information containing the 'getUrl'.
   */
  public function getPreviewUrl(string $driveId, string $itemId): array {
    $endpoint = "/drives/{$driveId}/items/{$itemId}/preview";
    // Preview URL generation requires a POST request. [4]
    return $this->makeRequest('POST', $endpoint, ['json' => []]);
  }

  /**
   * Gets a short-lived, pre-authenticated download URL for a file.
   *
   * @param string $driveId
   *   The ID of the drive containing the item.
   * @param string $itemId
   *   The ID of the file item.
   *
   * @return string|null
   *   The download URL, or null on failure.
   */
  public function getDownloadUrl(string $driveId, string $itemId):?string {
    // Select the special @microsoft.graph.downloadUrl property. [5]
    $safeDrive = rawurlencode($driveId);
    $safeItem = rawurlencode($itemId);
    $endpoint = "/drives/{$safeDrive}/items/{$safeItem}?\$select=id,@microsoft.graph.downloadUrl";
    $response = $this->makeRequest('GET', $endpoint);
    return $response['@microsoft.graph.downloadUrl'] ?? NULL;
  }

  /**
   * Get item metadata (e.g., name) for a drive item.
   */
  public function getItem(string $driveId, string $itemId): array {
    $safeDrive = rawurlencode($driveId);
    $safeItem = rawurlencode($itemId);
    $endpoint = "/drives/{$safeDrive}/items/{$safeItem}?\$select=id,name";
    return $this->makeRequest('GET', $endpoint);
  }

  /**
   * Opens a streaming response for a file's content.
   *
   * @return array{stream:\Psr\Http\Message\StreamInterface, headers:array}|array
   *   Returns array with 'stream' and 'headers' on success; empty array on failure.
   */
  public function openContentStream(string $driveId, string $itemId): array {
    try {
      $token = $this->tokenFetcher->fetchToken('client_credentials', [
        'scope' => 'https://graph.microsoft.com/.default',
      ]);
      $safeDrive = rawurlencode($driveId);
      $safeItem = rawurlencode($itemId);
      $url = self::GRAPH_API_BASE_URL . "/drives/{$safeDrive}/items/{$safeItem}/content";
      $res = $this->httpClient->request('GET', $url, [
        'headers' => [
          'Authorization' => 'Bearer ' . $token,
        ],
        'stream' => TRUE,
        'allow_redirects' => TRUE,
      ]);
      return [
        'stream' => $res->getBody(),
        'headers' => $res->getHeaders(),
      ];
    }
    catch (RequestException $e) {
      $this->exceptionLogger->handleException($e);
    }
    catch (\Exception $e) {
      $this->exceptionLogger->handleException($e);
    }
    return [];
  }

  /**
   * Searches for items within a specific drive.
   *
   * @param string $driveId
   *   The ID of the drive to search in.
   * @param string $searchText
   *   The text to search for.
   *
   * @return array
   *   An array of drive items that match the search query.
   */
  public function searchDrive(string $driveId, string $searchText): array {
    // The search text needs to be properly encoded for the URL.
    $encodedSearchText = rawurlencode($searchText);
    $endpoint = "/drives/{$driveId}/root/search(q='{$encodedSearchText}')";
    return $this->makeRequest('GET', $endpoint);
  }

  /**
   * Gets changes for a drive using a delta token.
   *
   * @param string $driveId
   *   The ID of the drive to get changes for.
   * @param string|null $deltaToken
   *   (Optional) The delta token from a previous API call. If null, a full
   *   sync is initiated.
   *
   * @return array
   *   An array of changed items and a new deltaLink/deltaToken.
   */
  public function getDriveChanges(string $driveId, string $deltaToken = NULL): array {
    if ($deltaToken) {
      // If we have a token, we call the delta URL directly. [6]
      // The token itself is a full URL.
      return $this->makeRequest('GET', '', ['base_uri' => $deltaToken]);
    }
    else {
      // For the initial request, we call the delta endpoint.
      $endpoint = "/drives/{$driveId}/root/delta";
      return $this->makeRequest('GET', $endpoint);
    }
  }

  /**
   * Retrieves the permissions for a specific file or folder.
   *
   * @param string $driveId
   *   The ID of the drive.
   * @param string $itemId
   *   The ID of the item (file or folder).
   *
   * @return array
   *   An array of permission resources.
   */
  public function getItemPermissions(string $driveId, string $itemId): array {
    $endpoint = "/drives/{$driveId}/items/{$itemId}/permissions";
    return $this->makeRequest('GET', $endpoint);
  }

  /**
   * Retrieves the permissions for a specific site.
   *
   * @param string $siteId
   *   The ID of the site.
   *
   * @return array
   *   An array of permission resources.
   */
  public function getSitePermissions(string $siteId): array {
    $endpoint = "/sites/{$siteId}/permissions";
    return $this->makeRequest('GET', $endpoint);
  }

  /**
   * Grants new permissions (sends an invitation) for an item.
   *
   * @param string $driveId
   *   The ID of the drive.
   * @param string $itemId
   *   The ID of the item.
   * @param array $recipients
   *   An array of recipient email addresses.
   * @param string $role
   *   The role to grant ('read' or 'write').
   * @param bool $requireSignIn
   *   Whether the recipient must sign in.
   * @param bool $sendInvitation
   *   Whether to send an email invitation.
   * @param string $message
   *   A message to include in the invitation.
   *
   * @return array
   *   The response from the API.
   */
  public function grantItemPermission(string $driveId, string $itemId, array $recipients, string $role = 'read', bool $requireSignIn = TRUE, bool $sendInvitation = TRUE, string $message = 'Here is the file we discussed.'): array {
    $endpoint = "/drives/{$driveId}/items/{$itemId}/invite";
    $recipientData = array_map(function ($email) {
      return ['email' => $email];
    }, $recipients);

    $body = [
      'recipients' => $recipientData,
      'roles' => [$role],
      'requireSignIn' => $requireSignIn,
      'sendInvitation' => $sendInvitation,
      'message' => $message,
    ];

    return $this->makeRequest('POST', $endpoint, ['json' => $body]);
  }

  /**
   * Creates a new folder.
   *
   * @param string $driveId
   *   The ID of the drive.
   * @param string $parentFolderId
   *   The ID of the parent folder ('root' for the root directory).
   * @param string $folderName
   *   The name of the new folder.
   *
   * @return array
   *   The created driveItem resource.
   */
  public function createFolder(string $driveId, string $parentFolderId, string $folderName): array {
    $endpoint = "/drives/{$driveId}/items/{$parentFolderId}/children";
    $body = [
      'name' => $folderName,
      'folder' => new \stdClass(),
      '@microsoft.graph.conflictBehavior' => 'rename',
    ];
    return $this->makeRequest('POST', $endpoint, ['json' => $body]);
  }

  /**
   * Uploads a small file (under 4MB).
   *
   * @param string $driveId
   *   The ID of the drive.
   * @param string $parentFolderId
   *   The ID of the parent folder.
   * @param string $fileName
   *   The name of the file to create.
   * @param mixed $fileContent
   *   The binary content of the file.
   *
   * @return array
   *   The created driveItem resource.
   */
  public function uploadSmallFile(string $driveId, string $parentFolderId, string $fileName, $fileContent): array {
    $encodedFileName = rawurlencode($fileName);
    $endpoint = "/drives/{$driveId}/items/{$parentFolderId}:/{$encodedFileName}:/content";
    $options = ['body' => $fileContent];
    return $this->makeRequest('PUT', $endpoint, $options);
  }

  /**
   * Creates an upload session for a large file (over 4MB).
   *
   * @param string $driveId
   *   The ID of the drive.
   * @param string $parentFolderId
   *   The ID of the parent folder.
   * @param string $fileName
   *   The name of the file.
   *
   * @return array
   *   The upload session information, including the 'uploadUrl'.
   */
  public function createUploadSession(string $driveId, string $parentFolderId, string $fileName): array {
    $encodedFileName = rawurlencode($fileName);
    $endpoint = "/drives/{$driveId}/items/{$parentFolderId}:/{$encodedFileName}:/createUploadSession";
    $body = [
      'item' => [
        '@microsoft.graph.conflictBehavior' => 'rename',
      ],
    ];
    return $this->makeRequest('POST', $endpoint, ['json' => $body]);
  }

  /**
   * Helper function to make an authenticated request to the Graph API.
   *
   * @param string $method
   *   The HTTP method (e.g., 'GET', 'POST').
   * @param string $endpoint
   *   The API endpoint to call.
   * @param array $options
   *   Additional options for the Guzzle request.
   *
   * @return array
   *   The decoded JSON response.
   */
  private function makeRequest(string $method, string $endpoint, array $options = []): array {
    try {
      $token = $this->tokenFetcher->fetchToken('client_credentials', [
        'scope' => 'https://graph.microsoft.com/.default',
      ]);

      // If a full base_uri is passed in options, use it directly (for delta).
      $url = isset($options['base_uri']) ? $options['base_uri'] : self::GRAPH_API_BASE_URL . $endpoint;

      $default_options = [
        'headers' => [
          'Authorization' => 'Bearer ' . $token,
        ],
      ];

      // Ensure Content-Type is set for POST/PUT/PATCH if not already.
      if (in_array($method, ['POST', 'PUT', 'PATCH']) && !isset($options['headers']['Content-Type'])) {
        $default_options['headers']['Content-Type'] = 'application/json';
      }

      $request_options = array_merge_recursive($default_options, $options);

      $response = $this->httpClient->request($method, $url, $request_options);
      $contents = $response->getBody()->getContents();

      // Handle empty response body for certain successful requests (e.g., 204 No Content).
      if (empty($contents)) {
        return [];
      }

      $data = json_decode($contents, TRUE);

      if (isset($data['error'])) {
        $this->exceptionLogger->Exception(json_encode($data['error']));
      }
      return $data;
    }
    catch (RequestException $e) {
      $this->exceptionLogger->handleException($e);
    }
    catch (\Exception $e) {
      $this->exceptionLogger->handleException($e);
    }
    return [];
  }
}
