<?php

namespace Drupal\sharepoint_integration\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\StreamedResponse;
use Drupal\sharepoint_integration\API\GraphApiService;

class DownloadController extends ControllerBase implements ContainerInjectionInterface {

  /** @var \Drupal\sharepoint_integration\API\GraphApiService */
  protected $graphApiService;

  public function __construct(GraphApiService $graph_api_service) {
    $this->graphApiService = $graph_api_service;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('sharepoint_integration.graph_api_client')
    );
  }

  public function download(Request $request, string $driveId, string $itemId) {
    // Stable controller URL: no token validation. The controller itself
    // generates a fresh pre-authenticated Graph URL and streams the file.

    // Invalidate old copied links automatically if admin changes the selected
    // drive in module settings (free variant supports one public node).
    $currentDrive = \Drupal::config('sharepoint_integration.settings')->get('sharepoint_drive');
    if (!empty($currentDrive) && $currentDrive !== $driveId) {
      return $this->buildAccessDeniedResponse();
    }

    // Try to fetch a friendly filename by asking item metadata first.
    $name = $itemId;
    try {
      $meta = $this->graphApiService->getItem($driveId, $itemId);
      if (!empty($meta['name'])) {
        $name = $meta['name'];
      }
    }
    catch (\Exception $e) {}

    // Prefer streaming directly from Graph to avoid redirects/token issues.
    $graphStream = $this->graphApiService->openContentStream($driveId, $itemId);
    if (empty($graphStream)) {
      return $this->buildAccessDeniedResponse();
    }

    $stream = $graphStream['stream'];
    $filename = $name ?: ($itemId . '.bin');
    $response = new StreamedResponse(function () use ($stream) {
      while (!$stream->eof()) {
        echo $stream->read(8192);
        flush();
      }
    });
    $response->headers->set('Content-Type', 'application/octet-stream');
    $response->headers->set('Content-Disposition', 'attachment; filename="' . $filename . '"');
    $response->headers->set('Cache-Control', 'private, no-store, no-cache, must-revalidate');
    $response->headers->set('Pragma', 'no-cache');
    $response->headers->set('Expires', '0');
    return $response;
  }

  private function buildAccessDeniedResponse() {
    $response = new StreamedResponse(function () {
      echo 'Access denied.';
    });
    $response->setStatusCode(403);
    return $response;
  }
}


