<?php

namespace Drupal\sharepoint_integration\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\sharepoint_integration\SharepointItemInterface;

/**
 * Defines the SharePoint Item entity.
 *
 * This entity will store metadata for SharePoint sites, drives, folders, and
 * files, allowing them to be managed and displayed within Drupal.
 *
 * @ContentEntityType(
 * id = "sharepoint_item",
 * label = @Translation("SharePoint Item"),
 * base_table = "sharepoint_item",
 * entity_keys = {
 * "id" = "id",
 * "uuid" = "uuid",
 * "label" = "name",
 * },
 * handlers = {
 * "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 * "list_builder" = "Drupal\Core\Entity\EntityListBuilder",
 * "views_data" = "Drupal\views\EntityViewsData",
 * "access" = "Drupal\Core\Entity\EntityAccessControlHandler",
 * },
 * )
 */
class SharepointItem extends ContentEntityBase implements SharepointItemInterface {

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    // The human-readable name of the item (e.g., "Q3 Report.docx").
    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setDescription(t('The name of the SharePoint item.'))
      ->setRequired(TRUE)
      ->setSettings([
        'max_length' => 255,
      ])
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ]);

    // The unique ID from the Microsoft Graph API. This is critical.
    $fields['sharepoint_id'] = BaseFieldDefinition::create('string')
      ->setLabel(t('SharePoint ID'))
      ->setDescription(t('The unique identifier from the SharePoint API.'))
      ->setRequired(TRUE)
      ->addConstraint('UniqueField');

    // The type of item, used to differentiate and filter.
    $fields['item_type'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Item Type'))
      ->setDescription(t('The type of the SharePoint item.'))
      ->setRequired(TRUE)
      ->setSettings([
        'allowed_values' => [
          'site' => 'Site',
          'drive' => 'Drive',
          'folder' => 'Folder',
          'file' => 'File',
        ],
      ])
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'list_default',
        'weight' => 0,
      ]);

    // The direct URL to the item in SharePoint.
    $fields['web_url'] = BaseFieldDefinition::create('uri')
      ->setLabel(t('Web URL'))
      ->setDescription(t('The direct web URL to the item in SharePoint.'));

    // **NEW** Stores the SharePoint ID of the parent Drive (Document Library).
    // This is crucial for efficiently generating preview/download URLs.
    $fields['drive_id'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Drive ID'))
      ->setDescription(t('The SharePoint ID of the parent Drive (Document Library).'));

    // Stores the SharePoint ID of the direct parent item.
    $fields['parent_id'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Parent SharePoint ID'))
      ->setDescription(t('The SharePoint ID of the parent item.'))
      ->setRequired(FALSE);

    // An entity reference to the parent item within Drupal.
    $fields['parent_entity'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Parent Item'))
      ->setDescription(t('The parent SharePoint Item entity.'))
      ->setSetting('target_type', 'sharepoint_item');

    // Standard Drupal created timestamp.
    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    // Standard Drupal changed timestamp.
    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last updated.'));

    return $fields;
  }

}
