<?php

namespace Drupal\sharepoint_integration\EventSubscriber;

use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Drupal\Core\Config\ConfigCrudEvent;
use Drupal\Core\Config\ConfigEvents;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\node\Entity\Node;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\State\StateInterface;


/**
 * Class ConfigSubscriber.
 */
class ConfigSubscriber implements EventSubscriberInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The alias storage.
   *
   * @var \Drupal\path_alias\AliasStorageInterface
   * Constructs a ConfigSubscriber object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  protected $aliasStorage;

    public function __construct(ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager, StateInterface $state) {
        $this->configFactory = $config_factory;
        $this->entityTypeManager = $entity_type_manager;
        $this->state = $state;
    }

    /**
     * {@inheritdoc}
     */
    public static function getSubscribedEvents(): array {
        $events[ConfigEvents::SAVE][] = ['onConfigSave'];
        return $events;
    }

    /**
     *
     * @param \Drupal\Core\Config\ConfigCrudEvent $event
     *   The event to process.
     */
    public function onConfigSave(ConfigCrudEvent $event) {
        $config = $event->getConfig();

        if ($config->getName() !== 'sharepoint_integration.settings') {
          return;
        }

        // Only act when the drive markup changed to avoid re-entrancy/deadlocks
        // on unrelated settings saves.
        if (method_exists($event, 'isChanged')) {
          if (!$event->isChanged('sharepoint_current_drive_files')) {
            return;
          }
        }

        $table_blob = $config->get('sharepoint_current_drive_files');
        if (empty($table_blob)) {
          return;
        }

        $this->createOrUpdateNodeWithAlias($config);
    }

    /**
     * Creates or updates a node and sets an alias.
     *
     * @param \Drupal\Core\Config\ImmutableConfig $config
     *   The configuration object.
     */
    protected function createOrUpdateNodeWithAlias($config) {
        $table_markup = $config->get('sharepoint_current_drive_files') ? json_decode(gzuncompress($config->get('sharepoint_current_drive_files'))) : '';

        $node_storage = $this->entityTypeManager->getStorage('node');
        $nodes = $node_storage->loadByProperties(['title' => 'MO SharePoint']);

        if (!empty($nodes)) {
        $node = reset($nodes);
        $node->set('body', [
            'value' => '<div id="files-wrapper">' . $table_markup . '</div>',
            'format' => 'full_html',
        ]);
        } else {
        $node = Node::create([
            'type' => 'page',
            'title' => 'MO SharePoint',
            'body' => [
            'value' => '<div id="files-wrapper">' . $table_markup . '</div>',
            'format' => 'full_html',
            ],
            'status' => 1,
        ]);
        }
        $node->save();

        // Track created/updated node ID without triggering config save loops.
        $this->state->set('sharepoint_integration.mo_sharepoint_node_id', $node->id());

    }
}