<?php

namespace Drupal\sharepoint_integration\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\sharepoint_integration\API\TokenFetcher;
use Drupal\sharepoint_integration\API\GraphApiService;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Url;
use Drupal\sharepoint_integration\Utilities\MOUtilities;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;


/**
 * Configure SharePoint Integration settings for this site.
*/
class Connection extends ConfigFormBase {

  /**
   * @var \Drupal\sharepoint_integration\API\TokenFetcher
   */
  protected $tokenFetcher;

  /**
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * @var \Drupal\sharepoint_integration\API\GraphApiService
   */
  protected $graphApiService;

  /**
   * @var \Drupal\sharepoint_integration\Utilities\MOUtilities
   */
  protected $moUtilities;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;


  /**
   * Constructs a SharePointSettingsForm object.
   *
   * @param \Drupal\sharepoint_integration\API\TokenFetcher $token_fetcher
   *   The token fetcher service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\sharepoint_integration\API\GraphApiService $graph_api_service
   *   The graph api service.
   * @param \Drupal\sharepoint_integration\Utilities\MOUtilities $mo_utilities
   *   The mo utilities service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(TokenFetcher $token_fetcher, MessengerInterface $messenger, GraphApiService $graph_api_service, MOUtilities $mo_utilities, EntityTypeManagerInterface $entity_type_manager) {
    $this->tokenFetcher = $token_fetcher;
    $this->messenger = $messenger;
    $this->graphApiService = $graph_api_service;
    $this->moUtilities = $mo_utilities;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('sharepoint_integration.token_fetcher'),
      $container->get('messenger'),
      $container->get('sharepoint_integration.graph_api_client'),
      $container->get('sharepoint_integration.mo_utilities'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'sharepoint_integration_connection';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['sharepoint_integration.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('sharepoint_integration.settings');

    $form['#attached']['library'][] = 'sharepoint_integration/sharepoint_integration';


    // Stay on the same page on submit.
    unset($form['#attributes']['target']);

    $form['description'] = [
      '#type' => 'item',
      '#markup' => $this->t('Create an Azure AD App registration to securely connect Drupal with your SharePoint content.'),
    ];

    $form['application_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Application ID'),
      '#default_value' => $config->get('application_id'),
      '#placeholder' => $this->t('Enter the Application (Client) ID.'),
      '#description' => $this->t('<b>Note:</b> The Application ID can be found in the Active Directory application\'s Overview tab.'),
      '#required' => TRUE,
    ];

    $form['client_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client Secret'),
      '#default_value' => $config->get('client_secret'),
      '#placeholder' => $this->t('Enter the Client Secret.'),
      '#description' => $this->t('<b>Note:</b> The Client Secret value is present in the Active Directory application\'s Certificates & Secrets tab.'),
      '#required' => TRUE,
    ];

    $form['tenant_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tenant ID'),
      '#default_value' => $config->get('tenant_id'),
      '#placeholder' => $this->t('Enter the Directory (Tenant) ID.'),
      '#description' => $this->t('<b>Note:</b> The Tenant ID is present in the Active Directory application\'s Overview tab.'),
      '#required' => TRUE,
    ];

    // Removed cron-based sync; listings are rendered live.


    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save configs'),
      '#button_type' => 'primary',
    ];

    $form['test_connection'] = [
      '#type' => 'submit',
      '#value' => $this->t('Test connection'),
      '#submit' => ['::testConnection'],
      '#disabled' => !(bool) ($config->get('application_id') && $config->get('client_secret') && $config->get('tenant_id')),
      '#description' => $this->t('Save configs before testing the connection.'),
    ];

    // Removed manual full sync button.

    $this->moUtilities->showCustomerSupportIcon($form, $form_state);

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->configFactory->getEditable('sharepoint_integration.settings')
      ->set('application_id', $form_state->getValue('application_id'))
      ->set('client_secret', $form_state->getValue('client_secret'))
      ->set('tenant_id', $form_state->getValue('tenant_id'))
      ->save();

    parent::submitForm($form, $form_state);
  }


  public function testConnection() {
    try {
      $sitesData = $this->graphApiService->getSites();
      if (!empty($sitesData['value'])) {
        // Build and cache site options for the Preview tab.
        $options = [];
        foreach ($sitesData['value'] as $site) {
          if (!empty($site['id']) && !empty($site['displayName'])) {
            $options[$site['id']] = $site['displayName'];
          }
        }
        // Store in config so Preview can populate the dropdown without re-calling Graph.
        // Using a different key than drive files so the subscriber won't run.
        $this->configFactory->getEditable('sharepoint_integration.settings')
          ->set('sharepoint_site_options', gzcompress(json_encode($options)))
          ->save();

        $this->messenger->addStatus($this->t('Successfully connected to SharePoint. Found @count site(s).', ['@count' => count($options)]));
        \Drupal::logger('sharepoint_integration')->info('Successfully tested connection and fetched sites.');
      }
      else {
        $this->messenger->addError($this->t('Failed to connect to SharePoint or no sites were found. Please check the logs for more details.'));
      }
    }
    catch (\Exception $e) {
      $this->messenger->addError($this->t('An unexpected error occurred: @message', ['@message' => $e->getMessage()]));
    }
  }

  // Full sync functionality removed.

}
