<?php

namespace Drupal\sharepoint_integration\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\sharepoint_integration\API\TokenFetcher;
use Drupal\sharepoint_integration\API\GraphApiService;
use Drupal\sharepoint_integration\Utilities\MOUtilities;
use Drupal\Core\Url;
use Drupal\sharepoint_integration\Utilities\TableGenerator;


class PreviewContent extends FormBase {

  protected $tokenFetcher;
  protected $graphAPIClient;
  protected $tableGenerator;
  protected $moUtilities;


  public function __construct(TokenFetcher $token_fetcher, GraphApiService $graph_api_client,TableGenerator $table_generator, MOUtilities $mo_utilities) {
    $this->tokenFetcher = $token_fetcher;
    $this->graphAPIClient = $graph_api_client;
    $this->tableGenerator = $table_generator;
    $this->moUtilities = $mo_utilities;

  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('sharepoint_integration.token_fetcher'),
      $container->get('sharepoint_integration.graph_api_client'),
      $container->get('sharepoint_integration.table_generator'),
      $container->get('sharepoint_integration.mo_utilities')

    );
  }

  public function getFormId() {
    return 'sharepoint_integration_preview_content_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['sharepoint_integration.settings'];
  }

  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('sharepoint_integration.settings');
    $form['#attached']['library'][] = 'sharepoint_integration/sharepoint_integration';
    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';
    $this->moUtilities->showCustomerSupportIcon($form, $form_state);
    if(empty($config->get('client_secret'))){
      $form['description'] = [
        '#type' => 'item',
        '#markup' => $this->t('Please provide the Client ID, Secret, and Tenant ID of the Azure App Registration on the <a href="'.Url::fromRoute('sharepoint_integration.connection')->toString().'">Connection</a> tab.'),
      ];
      return $form;
    }
    try {
      // Removed Profile selector; keeping UI focused on Site and Drive.

      // Load sites list from cached options if available, else fetch.
      $cached = $config->get('sharepoint_site_options');
      $options = ['- Select -' => '- Select -'];
      if ($cached) {
        $decoded = json_decode(gzuncompress($cached), TRUE) ?: [];
        $options += $decoded;
      }
      else {
        $sites = $this->graphAPIClient->getSites();
        foreach ($sites['value'] as $site) {
          $options[$site['id']] = $site['displayName'];
        }
      }
      $request = \Drupal::request();
      $directoryId = $request->query->get('directoryId') ?? $config->get('sharepoint_drive');
      $parentFolderId = $request->query->get('parentFolderId');
      $form['sharepoint_site'] = [
        '#type' => 'select',
        '#title' => $this->t('Select SharePoint Site'),
        '#default_value' => $config->get('sharepoint_site'),
        '#options' => $options,
        '#attributes' => ['style' => 'width: 50%;'],
        '#prefix' => '<div id="sites-wrapper">',
        '#suffix' => '</div>',
        '#ajax' => [
          'callback' => '::loadDirectories',
          'wrapper' => 'directories-wrapper',
        ],
      ];
      $drive_options = $config->get('sharepoint_drive_options') ? json_decode(gzuncompress($config->get('sharepoint_drive_options'))) : [];
      $form['directories'] = [
        '#type' => 'select',
        '#title' => $this->t('Select Directory'),
        '#attributes' => ['style' => 'width: 50%;'],
        '#prefix' => '<div id="directories-wrapper">',
        '#default_value' => $directoryId,
        '#suffix' => '</div>',
        '#options' => $drive_options,
        '#validated' => TRUE,
        '#ajax' => [
          'callback' => '::loadFiles',
          'wrapper' => 'files-wrapper',
        ],
      ];
      // Add another drive (Premium) CTA directly below selects.
      $selectedSite = $form_state->getValue('sharepoint_site');
      $selectedDrive = $form_state->getValue('directories');
      $form['add_another_drive'] = [
        '#type' => 'link',
        '#title' => $this->t('Add another drive'),
        '#url' => Url::fromRoute('sharepoint_integration.multi_node_info'),
        '#attributes' => [
          'class' => ['button', 'button--primary', 'use-ajax'],
          'data-dialog-type' => 'modal',
          'data-dialog-options' => json_encode(['width' => '40%']),
        ],
        '#weight' => 2,
        '#access' => TRUE,
      ];

      // Public page URL (node) helper as info label with non-reloading copy button.
      $node_id = \Drupal::state()->get('sharepoint_integration.mo_sharepoint_node_id');
      if ($node_id) {
        $public_url = Url::fromUri('internal:/node/' . $node_id)->setAbsolute()->toString();
        $form['public_page'] = [
          '#type' => 'container',
          'label' => [
            '#type' => 'item',
            '#title' => $this->t('Public page URL'),
            '#markup' => '<a class="mo-info-url" href="' . $public_url . '" target="_blank" rel="noopener noreferrer">' . $public_url . '</a>',
          ],
          'copy' => [
            '#type' => 'button',
            '#value' => $this->t('Copy'),
            '#attributes' => [
              'class' => ['button', 'button--small', 'mo-copy-btn'],
              'onclick' => 'event.preventDefault(); navigator.clipboard.writeText("' . $public_url . '"); return false;',
              'type' => 'button',
            ],
          ],
          '#attributes' => ['class' => ['mo-inline-actions']],
          '#weight' => 9,
        ];
      }
      // Breadcrumb/back link logic (simple one-level up for now).
      $breadcrumb = '';
      if ($parentFolderId) {
        $back_url = Url::fromRoute('<current>', [], [
          'query' => [
            'directoryId' => $directoryId,
          ],
        ]);
        $breadcrumb = '<div class="mo-breadcrumbs"><a href="' . $back_url->toString() . '">' . $this->t('Drive root') . '</a></div>';
      }
      // Ensure an AJAX wrapper exists for the files table, so the first
      // selection can update without a page refresh.
      $form['files'] = [
        '#type' => 'markup',
        '#prefix' => '<div id="files-wrapper">',
        '#markup' => '',
        '#suffix' => '</div>',
        '#weight' => 10,
      ];

      // Initial auto-render: if a drive is selected (either from config or first drive), render and save node content.
      if ($directoryId) {
        $table_render_array = $this->tableGenerator->getTable($directoryId, $parentFolderId, [
          'show_preview' => TRUE,
          'link_mode' => 'direct',
        ]);
        $renderer = \Drupal::service('renderer');
        $table_markup_admin = $renderer->render($table_render_array);

        // Save end-user markup (no preview; internal) to create/update node.
        $enduser_render_array = $this->tableGenerator->getTable($directoryId, NULL, [
          'show_preview' => FALSE,
          'link_mode' => 'internal',
        ]);
        $enduser_markup = $renderer->render($enduser_render_array);
        $this->configFactory->getEditable('sharepoint_integration.settings')
          ->set('sharepoint_site', $form['sharepoint_site']['#default_value'] ?? $config->get('sharepoint_site'))
          ->set('sharepoint_drive', $directoryId)
          ->set('sharepoint_current_drive_files', gzcompress(json_encode($enduser_markup)))
          ->save();

        // Node feedback for admin as a separate status bar above table.
        $created_node_section = '';
        $node_id = \Drupal::state()->get('sharepoint_integration.mo_sharepoint_node_id');
        if ($node_id) {
          $view_url = Url::fromUri('internal:/node/' . $node_id)->toString();
          $edit_url = Url::fromUri('internal:/node/' . $node_id . '/edit')->toString();
          $created_node_section = '<div class="messages messages--status">' . $this->t('A public page has been created/updated:') . ' <a href="' . $view_url . '" target="_blank">' . $this->t('View node') . '</a> · <a href="' . $edit_url . '" target="_blank">' . $this->t('Edit node') . '</a></div>';
          $form['status_message'] = [
            '#type' => 'markup',
            '#markup' => $created_node_section,
            '#weight' => 8,
          ];
        }
        // Replace the placeholder wrapper with the rendered table content.
        $form['files']['#markup'] = ($breadcrumb ? $breadcrumb . '<br><br>' : '') . $table_markup_admin;
      }

      // (Button already rendered above selects.)

      return $form;
    } catch (\Exception $e) {
      \Drupal::logger('sharepoint_integration')->error('Error fetching SharePoint sites: ' . $e->getMessage());
      $form['error']=[
        '#markup' => $this->t('Error fetching SharePoint sites.'),
      ];
      return $form;
    }
  }

  public function loadDirectories(array &$form, FormStateInterface $form_state) {
    try {
      $siteId = $form_state->getValue('sharepoint_site');
      $directories = $this->graphAPIClient->getDrives($siteId);
      $options = ['- Select -' => '- Select -'];
      foreach ($directories['value'] as $directory) {
        $options[$directory['id']] = $directory['name'];
      }
      $form['directories']['#options'] = $options;
      $this->configFactory->getEditable('sharepoint_integration.settings')
      ->set('sharepoint_site', $siteId)
      ->set('sharepoint_drive_options', gzcompress(json_encode($options)))
      ->clear('sharepoint_current_drive_files')
      ->save();
      return $form['directories'];
    } catch (\Exception $e) {
      \Drupal::logger('sharepoint_integration')->error('Error fetching directories: ' . $e->getMessage());
      return [
        '#markup' => $this->t('Error fetching directories.'),
      ];
    }
  }

  public function loadFiles(array &$form, FormStateInterface $form_state) {
    $directoryId = $form_state->getValue('directories');
    if (!$directoryId) { return ['#markup' => '']; }
    $renderer = \Drupal::service('renderer');
    // Admin table (with preview/direct links).
    $table_render_array = $this->tableGenerator->getTable($directoryId, NULL, [
      'show_preview' => TRUE,
      'link_mode' => 'direct',
    ]);
    $table_markup_admin = $renderer->render($table_render_array);
    // End-user markup to store in node content.
    $enduser_render_array = $this->tableGenerator->getTable($directoryId, NULL, [
      'show_preview' => FALSE,
      'link_mode' => 'internal',
    ]);
    $enduser_markup = $renderer->render($enduser_render_array);
    $this->configFactory->getEditable('sharepoint_integration.settings')
      ->set('sharepoint_drive', $directoryId)
      ->set('sharepoint_current_drive_files', gzcompress(json_encode($enduser_markup)))
      ->save();

    $form['files'] = [
      '#type' => 'markup',
      '#markup' => '<div id="files-wrapper">' . $table_markup_admin . '</div>',
      '#weight' => 10,
    ];
    return $form['files'];
  }

  public function loadSites(array &$form, FormStateInterface $form_state) {
    // For now, profile does not change credentials; just reload sites list.
    try {
      $cached = $this->config('sharepoint_integration.settings')->get('sharepoint_site_options');
      $options = ['- Select -' => '- Select -'];
      if ($cached) {
        $decoded = json_decode(gzuncompress($cached), TRUE) ?: [];
        $options += $decoded;
      }
      else {
        $sites = $this->graphAPIClient->getSites();
        foreach ($sites['value'] as $site) {
          $options[$site['id']] = $site['displayName'];
        }
      }
      $form['sharepoint_site']['#options'] = $options;
      return $form['sharepoint_site'];
    }
    catch (\Exception $e) {
      return ['#markup' => $this->t('Error loading sites.')];
    }
  }

  public function fetchDriveSubmit(array &$form, FormStateInterface $form_state) { }

  public function submitForm(array &$form, FormStateInterface $form_state) { }

}
