<?php
declare(strict_types=1);

namespace Drupal\sharepoint_integration\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\sharepoint_integration\Utilities\MOSupport;

final class SharePointTrialForm extends FormBase {

  public function getFormId(): string {
    return 'sharepoint_integration_trial_form';
  }

  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['#prefix'] = '<div id="modal_support_form">';
    $form['#suffix'] = '</div>';

    $form['status_messages'] = [
      '#type' => 'status_messages',
      '#weight' => -10,
    ];

    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';

    $user_email = (string) \Drupal::currentUser()->getEmail();

    $form['trial_email'] = [
      '#type' => 'email',
      '#title' => $this->t('Email'),
      '#default_value' => $user_email,
      '#required' => TRUE,
      '#attributes' => [
        'placeholder' => $this->t('Enter your email'),
        'style' => 'width:99%;margin-bottom:1%;',
      ],
    ];

    $form['trial_phone'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Phone number'),
      '#required' => FALSE,
      '#attributes' => [
        'placeholder' => $this->t('Enter your phone number'),
      ],
    ];

    $form['trial_query'] = [
      '#type' => 'textarea',
      '#required' => TRUE,
      '#title' => $this->t('Specify your use case'),
      '#attributes' => [
        'placeholder' => $this->t('Describe your requirement or use case here'),
        'style' => 'width:99%'
      ],
      '#rows' => 2,
    ];

    $form['trial_features'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Features you are interested in'),
      '#options' => [
        'seamless_streaming' => $this->t('Seamless Streaming'),
        'unified_access' => $this->t('Unified Access'),
        'flexible_embedding' => $this->t('Flexible Embedding'),
        'public_document_access' => $this->t('Public Document Access'),
        'role_based_folder_restriction' => $this->t('Role-Based Folder Restriction'),
        'document_upload' => $this->t('Document Upload'),
        'dynamic_content_integration' => $this->t('Dynamic Content Integration'),
        'centralized_management' => $this->t('Centralized Management'),
        'enhanced_security' => $this->t('Enhanced Security'),
      ],
      '#description' => $this->t('Select one or more features.'),
    ];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['send'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
      '#attributes' => [
        'class' => [
          'use-ajax',
          'button--primary',
        ],
      ],
      '#ajax' => [
        'callback' => [$this, 'submitModalFormAjax'],
        'event' => 'click',
      ],
    ];

    return $form;
  }

  public function submitModalFormAjax(array $form, FormStateInterface $form_state): AjaxResponse {
    $response = new AjaxResponse();

    if ($form_state->hasAnyErrors()) {
      $response->addCommand(new ReplaceCommand('#modal_support_form', $form));
      return $response;
    }

    $values = $form_state->getValues();
    $email = (string) ($values['trial_email'] ?? '');
    $phone = (string) ($values['trial_phone'] ?? '');
    $use_case = trim((string) ($values['trial_query'] ?? ''));

    $selected = array_filter($values['trial_features'] ?? []);
    $labels = [];
    if (!empty($selected) && isset($form['trial_features']['#options']) && is_array($form['trial_features']['#options'])) {
      foreach (array_keys($selected) as $key) {
        if (isset($form['trial_features']['#options'][$key])) {
          $labels[] = (string) $form['trial_features']['#options'][$key];
        }
      }
    }

    $features_text = !empty($labels) ? implode(', ', $labels) : '';
    $query = (!empty($features_text) ? 'Interested Features: ' . $features_text . '. ' : '') . 'Use case: ' . $use_case;

    $support = new MOSupport($email, $phone, $query, 'Trial Request');
    $ok = $support->sendSupportQuery();

    if ($ok) {
      $this->messenger()->addStatus($this->t('Trial request sent. We will get back to you shortly on @email.', ['@email' => $email]));
    }
    else {
      $this->messenger()->addError($this->t('Error while sending trial request. Please email <a href=":mail">:mail</a>.', [':mail' => 'drupalsupport@xecurify.com']));
    }

    $response->addCommand(new RedirectCommand(Url::fromRoute('sharepoint_integration.connection')->toString()));
    return $response;
  }

  public function submitForm(array &$form, FormStateInterface $form_state): void {}
}


