<?php

namespace Drupal\sharepoint_integration\Form;

use Drupal\Core\Url;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;

/**
 * SharePoint Integration Upgrade Plan form.
 */
class SharePointUpgradePlan extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'sharepoint-integration-upgrade-plan';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    global $base_url;
    $module_path = $base_url . '/' . \Drupal::service('extension.list.module')->getPath("sharepoint_integration");

    // Top 6 features from the public page.
    // Source: https://plugins.miniorange.com/drupal-microsoft-sharepoint-integration#features
    $features = [
      [
        'title' => 'Seamless Streaming',
        'description' => 'Securely stream SharePoint videos into your Drupal site without exposing download links for efficient and safe video sharing.',
      ],
      [
        'title' => 'Unified Access',
        'description' => 'Integrate SharePoint or OneDrive for Business directly into Drupal to eliminate platform switching and simplify document retrieval.',
      ],
      [
        'title' => 'Flexible Embedding',
        'description' => 'Embed images, PDFs, and Word documents with configurable view or edit abilities based on user permissions.',
      ],
      [
        'title' => 'Public Document Access',
        'description' => 'Allow anonymous access to selected SharePoint documents to facilitate public content sharing without login.',
      ],
      [
        'title' => 'Role-Based Folder Restriction',
        'description' => 'Protect sensitive information by enforcing role-based restrictions on folders and files.',
      ],
      [
        'title' => 'Document Upload',
        'description' => 'Provide a dedicated upload interface in Drupal to sync files directly to SharePoint drives for streamlined collaboration.',
      ],
    ];

    $related_products = [
      [ 'title' => 'OAuth Client', 'description' => 'OAuth/OpenID Connect Client SSO (OAuth 2.0) module allows users residing at the OAuth Provider\'s side to login to your Drupal site. The module syncs with all OAuth/OpenID providers that conform to OAuth 2.0 or OpenID Connect 1.0 standards.', 'image' => 'OAuth-Client.webp', 'link' => 'https://plugins.miniorange.com/drupal-sso-oauth-openid-single-sign-on' ],
      [ 'title' => 'SAML SP', 'description' => 'miniOrange provides Drupal SAML SP as a Single Sign-On solution that allows you to login to your Drupal site using SAML 2.0 compliant Identity Provider credentials. You can easily configure the Identity Provider with your Drupal site by simply providing a metadata URL or metadata file.', 'image' => 'SAML-SP.webp', 'link' => 'https://plugins.miniorange.com/drupal-saml-single-sign-on-sso' ],
      [ 'title' => 'Two Factor Authentication', 'description' => 'Second-Factor Authentication (TFA) adds a second layer of security with an option to configure truly Passwordless Login. You can configure the module to send an OTP to your preferred mode of communication like phone/email, integrate with TOTP Apps like Google Authenticator or configure hardware token 2FA method.', 'image' => '2FA.webp', 'link' => 'https://plugins.miniorange.com/drupal-two-factor-authentication-2fa' ],
      [ 'title' => 'OTP Verification', 'description' => 'Drupal OTP Verification module verifies Email Address/Mobile Number of users by sending verification code(OTP) during registration. It eliminates the possibility of a user registering with invalid personal details (phone number or email) on the Drupal site.', 'image' => 'OTP.webp', 'link' => 'https://plugins.miniorange.com/drupal-otp-verification' ],
      [ 'title' => 'REST API Authentication', 'description' => 'Drupal REST & JSON API Authentication module secures your Drupal site APIs against unauthorized access by enforcing different authentication methods including Basic Authentication, API Key Authentication, JWT Authentication, Third-Party Provider Authentication, etc.', 'image' => 'REST-API.webp', 'link' => 'https://plugins.miniorange.com/drupal-rest-api-authentication' ],
      [ 'title' => 'Website Security Pro', 'description' => 'The Website Security Pro module safeguards your Drupal site with enterprise-grade security. It protects against brute force and DDoS attacks, enforces strong passwords, monitors and blacklists suspicious IPs, and secures login and registration forms. Designed to block hackers and malware, it ensures your site stays secure, stable, and reliable.', 'image' => 'Web-Security.webp', 'link' => 'https://plugins.miniorange.com/drupal-web-security-pro' ],
    ];

    // Map to Twig variables expected by templates.
    $feature_items = [];
    foreach ($features as $feature) {
      $feature_items[] = [
        'title' => $feature['title'],
        'body' => $feature['description'],
        'url' => Url::fromUri('https://plugins.miniorange.com/drupal-sharepoint-integration')->toString(),
      ];
    }

    $product_items = [];
    foreach ($related_products as $product) {
      $product_items[] = [
        'title' => $product['title'],
        'body' => $product['description'],
        'image' => Markup::create('<img class="feature-image" src="' . $module_path . '/includes/images/' . $product['image'] . '" alt="' . $product['title'] . '" />'),
        'url' => $product['link'],
      ];
    }

    $steps = [
      $this->t('Click on Upgrade Now and sign in to the miniOrange console.'),
      $this->t('Proceed to the payment page and complete the purchase.'),
      $this->t('Download the licensed module(s) from the Releases and Downloads section.'),
      $this->t('Uninstall and delete the free version of the module.'),
      $this->t('Install the downloaded licensed version.'),
      Markup::create($this->t('Clear Drupal cache from the <a href=":url" target="_blank" rel="noopener noreferrer">Performance page</a>.', [
        ':url' => Url::fromRoute('system.performance_settings')->toString(),
      ])),
      $this->t('Enable the licensed module and log in with your miniOrange account.'),
    ];

    // Limit features to six items for the grid row layout.
    $feature_items = array_slice($feature_items, 0, 6);

    // Attach required libraries and pass variables to Twig.
    $form['#attached']['library'][] = 'sharepoint_integration/upgrade_plan';
    // Base module styles/scripts (positions the support icon, etc.).
    $form['#attached']['library'][] = 'sharepoint_integration/sharepoint_integration';
    $form['#theme'] = 'sharepoint_upgrade_plan';
    // External Upgrade Plan page.
    $form['#upgrade_url'] = Url::fromUri('https://plugins.miniorange.com/drupal-microsoft-sharepoint-integration')->toString();
    // View all features URL (as requested).
    $form['#features_all_url'] = Url::fromUri('https://plugins.miniorange.com/drupal-microsoft-sharepoint-integration#features')->toString();
    $form['#current_plan'] = 'Free Version';
    $form['#features'] = $feature_items;
    $form['#products'] = $product_items;
    $form['#steps'] = $steps;
    $form['#products_all_url'] = Url::fromUri('https://plugins.miniorange.com/drupal')->toString();

    // Keep customer support icon.
    $moUtilities = \Drupal::service('sharepoint_integration.mo_utilities');
    $moUtilities->showCustomerSupportIcon($form, $form_state);

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // No form submission needed for this display form
  }
}
