<?php

namespace Drupal\sharepoint_integration\Utilities;

use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\sharepoint_integration\API\TokenFetcher;
use Drupal\sharepoint_integration\API\GraphApiService;
use Drupal\Core\Render\Markup;
use Drupal\Core\Url;

class TableGenerator {

  protected $tokenFetcher;
  protected $graphAPIClient;
  public function __construct(TokenFetcher $token_fetcher, GraphApiService $graph_api_client) {
    $this->tokenFetcher = $token_fetcher;
    $this->graphAPIClient = $graph_api_client;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('sharepoint_integration.token_fetcher'),
      $container->get('sharepoint_integration.graph_api_client')
    );
  }

  public function getTable($directoryId, $parentFolderId = null, array $options = []) {
    $show_preview = $options['show_preview'] ?? TRUE;
    $link_mode = $options['link_mode'] ?? 'internal';

    $header = [
        'name'         => t('File/Folder Name'),
        'lastModified' => t('Last Modified'),
        'size'         => t('Size'),
        'downloadLink' => t('Download Link'),
    ];
    if ($show_preview) {
      $header['previewLink'] = t('Preview');
    }

    // Use GraphApiService wrapper to fetch items from drive/folder.
    $folderId = $parentFolderId ? (string) $parentFolderId : 'root';
    $data = $this->graphAPIClient->getDriveItems((string) $directoryId, $folderId);
    $files = [];
    if (isset($data['value'])) {
      foreach ($data['value'] as $item) {
        $is_folder = isset($item['folder']);
        $file = [
          'id'                   => $item['id'],
          'name'                 => $item['name'],
          'createdDateTime'      => $item['createdDateTime'],
          'lastModifiedDateTime' => $item['lastModifiedDateTime'],
          'webUrl'               => isset($item['webUrl']) ? $item['webUrl'] : '',
          'size'                 => isset($item['size']) ? $item['size'] : '',
          'mimeType'             => isset($item['file']['mimeType']) ? $item['file']['mimeType'] : '',
          'downloadUrl'          => isset($item['@microsoft.graph.downloadUrl']) ? $item['@microsoft.graph.downloadUrl'] : '',
          'is_folder'            => $is_folder,
        ];
        $files[] = $file;
      }
    }

    $rows = [];
    foreach ($files as $file) {
      if ($file['is_folder']) {
        // Render folder name as a link for drill-down (using query param for folderId).
        $url = Url::fromRoute('<current>', [], [
          'query' => [
            'directoryId'   => $directoryId,
            'parentFolderId'=> $file['id'],
          ],
        ]);
        $name = Markup::create('<a href="' . $url->toString() . '">' . htmlspecialchars($file['name']) . '</a>');
        $download = ['#markup' => t('N/A')];
        $preview = ['#markup' => t('N/A')];
      } else {
        $name = $file['name'];
        if (!empty($file['id'])) {
          // Stable controller URL without token. Controller will fetch fresh
          // pre-authenticated URL and stream the file.
          $url = Url::fromRoute('sharepoint_integration.download', [
            'driveId' => $directoryId,
            'itemId' => $file['id'],
          ]);
          $download = Markup::create('<a class="button button--small mo-action-btn" href="' . $url->toString() . '">' . t('Download') . '</a>');
        }
        else {
          $download = ['#markup' => t('N/A')];
        }
        if ($show_preview) {
          // Premium CTA for preview feature.
          $modalUrl = Url::fromRoute('sharepoint_integration.preview_info');
          $preview = Markup::create('<a class="button button--small mo-action-btn use-ajax" data-dialog-type="modal" data-dialog-options="{\"width\":\"40%\"}" href="' . $modalUrl->toString() . '">' . t('Preview') . '</a>');
        }
      }
      $row = [
        'name'         => $name,
        'lastModified' => $this->formatDate($file['lastModifiedDateTime']),
        'size'         => isset($file['size']) && $file['size'] !== '' ? $this->formatBytes((int) $file['size']) : t('—'),
        'downloadLink' => $download,
      ];
      if ($show_preview) {
        $row['previewLink'] = $preview;
      }
      $rows[] = $row;
    }
    return [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => t('No files/folders found.'),
    ];
  }

  private function formatBytes(int $bytes): string {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $i = 0;
    while ($bytes >= 1024 && $i < count($units) - 1) {
      $bytes /= 1024;
      $i++;
    }
    $precision = $i === 0 ? 0 : 1;
    return sprintf('%s %s', round($bytes, $precision), $units[$i]);
  }

  /**
   * Recursively fetch all files and folders from a SharePoint document library.
   *
   * @param string $directoryId
   *   The SharePoint drive/document library ID.
   * @param string|null $parentFolderId
   *   The parent folder ID, or null for root.
   * @param array $results
   *   (Internal) The results array, used for recursion.
   *
   * @return array
   *   Flat array of all items (files and folders) at all levels.
   */
  public function getAllItemsRecursive($directoryId, $parentFolderId = null, &$results = []) {
    $folderId = $parentFolderId ? (string) $parentFolderId : 'root';
    $data = $this->graphAPIClient->getDriveItems((string) $directoryId, $folderId);

    if (isset($data['value'])) {
      foreach ($data['value'] as $item) {
        $results[] = $item;
        if (isset($item['folder'])) {
          // Recursive call for subfolders
          $this->getAllItemsRecursive($directoryId, $item['id'], $results);
        }
      }
    }
    return $results;
  }

  private function formatDate($iso): string {
    if (empty($iso)) {
      return t('—');
    }
    try {
      $timestamp = strtotime($iso);
      if (!$timestamp) {
        return (string) $iso;
      }
      /** @var \Drupal\Core\Datetime\DateFormatterInterface $formatter */
      $formatter = \Drupal::service('date.formatter');
      return $formatter->format($timestamp, 'custom', 'Y-m-d H:i');
    }
    catch (\Exception $e) {
      return (string) $iso;
    }
  }
}