/**
 * @file
 * A Backbone View that controls the overall Simple hierarchical select widgets.
 *
 * @see Drupal.shs.AppModel
 */

(function ($, _, Backbone, Drupal, once) {

  'use strict';

  Drupal.shs.AppView = Backbone.View.extend(/** @lends Drupal.shs.AppView# */{
    /**
     * Container element for SHS widgets.
     */
    container: null,
    /**
     * Field configuration.
     *
     * @type {object}
     */
    config: {},
    /**
     * @constructs
     *
     * @augments Backbone.View
     *
     * @param {object} options
     *   An object with the following keys:
     * @param {Drupal.shs.AppModel} options.model
     *   The application state model.
     */
    initialize: function (options) {
      // Track app state.
      this.config = this.model.get('config');

      // Initialize collection.
      this.collection = new Drupal.shs.ContainerCollection();
      this.collection.reset();

      // Initialize event listeners.
      this.listenTo(this.collection, 'initialize:shs', this.renderWidgets);

      $(once("shs", this.$el)).addClass('hidden');
    },
    /**
     * Main render function of Simple hierarchical select.
     *
     * @return {Drupal.shs.AppView}
     *   Returns AppView for chaining.
     */
    render: function () {
      var app = this;

      // Create application container.
      app.container = $('<div>')
              .addClass('shs-container')
              .html('')
              .insertBefore(app.$el);

      // Generate widget containers.
      $.each(app.getConfig('parents'), function (delta, parents) {
        app.collection.add(new Drupal.shs.classes[app.getConfig('fieldName')].models.container({
          delta: delta,
          parents: parents
        }));
      });

      app.collection.trigger('initialize:shs');

      return app;
    },
    /**
     * Renders the select widgets of Simple hierarchical select.
     *
     * @return {Drupal.shs.AppView}
     *   Returns AppView for chaining.
     */
    renderWidgets: function () {
      var app = this;
      var fieldName = app.getConfig('fieldName');
      // Create widget containers.
      app.collection.each(function (containerModel) {
        var container = new Drupal.shs.classes[fieldName].views.container({
          app: app,
          model: containerModel
        });

        app.container.append(container.render().$el);
      });
      // Create button for "Add new".
      new Drupal.shs.classes[fieldName].views.addNew({
        app: app
      });

      app.collection.trigger('widgetsRendered:shs');
      return app;
    },
    /**
     * Update the value of the original element.
     *
     * @param {string} value
     *   New value of element.
     * @param {Drupal.shs.ContainerView} container
     *   Updated container.
     * @param {Drupal.shs.WidgetModel} widgetModel
     *   The changed model.
     */
    updateElementValue: function(value, container, widgetModel) {
      var app = this;

      if (app.getSetting('multiple')) {
        value = [];
        app.collection.each(function (model) {
          var modelValue = model.get('value');
          if (typeof modelValue == undefined || null == modelValue || modelValue === app.getSetting('anyValue')) {
            return;
          }
          value.push(modelValue);
        });
      }
      else {
        if (value === app.getSetting('anyValue') && widgetModel.get('level') > 0) {
          // Use value of parent widget (which is the id of the model ;)).
          value = widgetModel.get('id');
        }
        else if (widgetModel.get('createValue')) {
          // Add the created item to the original select item.
          var options = $("option", app.$el).map(function () {
            return $(this).val();
          }).get();
          if ($.inArray(value, options) === -1) {
            var item = widgetModel.get('createValue');
            app.$el.append($("<option/>").val(item.tid).text(item.name));
            // We can now reset our widget model to the new tid.
              widgetModel.set('createValue', null);
            widgetModel.set('defaultValue', item.tid);
          }
        }
      }
      // Set the updated value.
      app.$el.val(value).trigger({
        type: 'change',
        shsContainer: container,
        shsWidgetModel: widgetModel
      });
      return app;
    },
    /**
     * Check if original widget reports an error.
     *
     * @returns {boolean}
     *   Whether there is something wrong with the original widget.
     */
    hasError: function () {
      return this.$el.hasClass('error');
    },
    /**
     * Get a configuration value for shs.
     *
     * @param {string} name
     *   Name of the configuration to get. To get the value of a nested
     *   configuration the names are concatenated by a dot (i.e.
     *   "display.animationSpeed").
     *
     * @returns {mixed}
     *   The value of the configuration or the complete configuration object if
     *   the name is empty.
     */
    getConfig: function (name) {
      if (typeof name == undefined || name == null) {
        return this.config || {};
      }

      var parts = name.split('.');
      var conf = this.config || {};
      for (var i = 0, len = parts.length; i < len; i++) {
        conf = conf[parts[i]];
      }
      if (typeof conf === undefined) {
        return;
      }
      return conf;
    },
    /**
     * Shortcut function for <code>getConfig('settings.*');</code>.
     *
     * @param {string} name
     *   Name of a setting to get. If empty, the entire settings will be
     *   returned.
     *
     * @returns {mixed}
     *   The value of the setting.
     */
    getSetting: function (name) {
      if (typeof name == undefined || name == null) {
        name = 'settings';
      }
      else {
        name = 'settings.' + name;
      }
      return this.getConfig(name);
    }
  });

  /**
   * @constructor
   *
   * @augments Backbone.Collection
   */
  Drupal.shs.ContainerCollection = Backbone.Collection.extend(/** @lends Drupal.shs.ContainerCollection */{
    /**
     * @type {Drupal.shs.ContainerModel}
     */
    model: Drupal.shs.ContainerModel
  });

}(jQuery, _, Backbone, Drupal, once));
