<?php

namespace Drupal\sibs_api_commerce\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\HttpFoundation\Request;
use Drupal\commerce_payment\Exception\PaymentGatewayException;

/**
 * Provides the Off-site Redirect payment gateway.
 *
 * @CommercePaymentGateway(
 *   id = "sibs_api_spg_commerce",
 *   label = "SIBS API SPG OFFSITE",
 *   display_label = "SIBS SPG",
 *   forms = {
 *     "offsite-payment" = "Drupal\sibs_api_commerce\Plugin\Commerce\PluginForm\SibsSPGCommerceForm",
 *   },
 * )
 */
class SibsSPGCommerce extends OffsitePaymentGatewayBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'redirect_method' => 'post',
      'spg_methods' => '',
      'transition_state' => '',
      'transition_machinename_accepted' => '',
      'transition_machinename_failed' => ''
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form['spg_methods'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Avaliable payment methods'),
      '#options' => [
        "REFERENCE" => $this->t("REFERENCE"),
        "CARD" => $this->t("CARD"),
        "MBWAY" => $this->t("MBWAY"),
      ],
      '#default_value' => $this->configuration['spg_methods'],
      '#required' => 1,
    ];

    $form['transition_state'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Transition order state after payment recived'),
      '#description' => $this->t('I am a developer and I konow the machine name of the order transition'),
      '#default_value' => $this->configuration['transition_state'],
    ];

    $form['transition_machinename_accepted'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Order transition machine name for completed payment'),
      '#default_value' => $this->configuration['transition_machinename_accepted'],
      '#states' => [
        'visible' => [
          ':input[name="configuration[sibs_api_spg_commerce][transition_state]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['transition_machinename_failed'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Order transition machine name for declined payment'),
      '#default_value' => $this->configuration['transition_machinename_failed'],
      '#states' => [
        'visible' => [
          ':input[name="configuration[sibs_api_spg_commerce][transition_state]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    $values = $form_state->getValue($form['#parents']);
    $this->configuration['spg_methods'] = $values['spg_methods'];
    $this->configuration['transition_state'] = $values['transition_state'];
    $this->configuration['transition_machinename_accepted'] = $values['transition_machinename_accepted'];
    $this->configuration['transition_machinename_failed'] = $values['transition_machinename_failed'];
  }

  /**
   * {@inheritdoc}
   */
  public function onReturn(OrderInterface $order, Request $request) {
    $payment_storage = \Drupal::entityTypeManager()->getStorage('commerce_payment');
    $payments = $payment_storage->loadMultipleByOrder($order);
    $payment = end($payments);
    $service = \Drupal::service('sibs_api.spg');
    $status = $service->getpaymentStatus($payment->remote_id->value);
    //dd($status);
    if ($status["paymentStatus"] == 'Declined' || $status["paymentStatus"] == 'Timeout') {
      throw new PaymentGatewayException("Failed");
    }
  }

}