(function (Drupal, drupalSettings) {
  'use strict';

  function withinBreakpoint(container) {
    var bp = parseInt(container.getAttribute('data-sn-breakpoint') || drupalSettings?.sidenotes?.breakpoint || 768, 10);
    return window.matchMedia('(max-width: ' + bp + 'px)').matches;
  }

  function buildEndnotes(container) {
    if (container.querySelector('.sn-endnotes')) {
      return; // Already built.
    }
    var endnotes = document.createElement('section');
    endnotes.className = 'sn-endnotes';
    var title = document.createElement('h2');
    title.className = 'sn-endnotes-title';
    var fallbackTitle = 'Notes';
    var configuredTitle = container.getAttribute('data-sn-endnotes-title');
    title.textContent = configuredTitle || fallbackTitle;
    var list = document.createElement('ol');
    list.className = 'sn-endnotes-list';
    endnotes.appendChild(title);
    endnotes.appendChild(list);

    var content = container.querySelector('.sn-content') || container;
    content.parentNode.insertBefore(endnotes, content.nextSibling);

    var notes = container.querySelectorAll('.sn-note');
    notes.forEach(function (note) {
      var idx = note.getAttribute('data-sn-index');
      var li = document.createElement('li');
      li.className = 'sn-endnote';
      li.id = 'sn-end-' + idx;

      var back = document.createElement('a');
      back.href = '#snref' + idx;
      back.className = 'sn-endnote-backref';
      back.setAttribute('aria-label', 'Back to reference');
      back.textContent = '↩';

      // Clone the note content without the number and without nested notes.
      var contentSpan = note.querySelector('.sn-note-content');
      var clone = contentSpan ? contentSpan.cloneNode(true) : document.createElement('span');
      li.appendChild(clone);
      li.appendChild(back);
      list.appendChild(li);
    });
  }

  function applyResponsiveBehavior(container) {
    var mode = container.getAttribute('data-sn-mobile') || 'inline';
    var isNarrow = withinBreakpoint(container);

    if (isNarrow) {
      container.classList.remove('sn-is-desktop');
      if (mode === 'endnotes') {
        buildEndnotes(container);
        container.classList.add('sn-is-mobile-endnotes');
        container.classList.remove('sn-is-mobile-inline');
      } else {
        container.classList.add('sn-is-mobile-inline');
        container.classList.remove('sn-is-mobile-endnotes');
        relocateInlineNotes(container);
      }
    } else {
      container.classList.remove('sn-is-mobile-inline');
      container.classList.remove('sn-is-mobile-endnotes');
      container.classList.add('sn-is-desktop');
      restoreInlineNotes(container);
    }
  }

  function ensurePlaceholder(note) {
    var id = note.getAttribute('id');
    if (!id) return null;
    var placeholderId = 'snph-' + id;
    var existing = note.parentNode ? note.parentNode.querySelector('#' + placeholderId) : null;
    if (existing) return existing;
    var ph = document.createElement('span');
    ph.className = 'sn-note-placeholder';
    ph.id = placeholderId;
    note.parentNode && note.parentNode.insertBefore(ph, note);
    return ph;
  }

  function relocateInlineNotes(container) {
    var pos = container.getAttribute('data-sn-inline-pos') || 'after_paragraph';
    if (pos !== 'after_paragraph') return; // Keep default immediate-after behavior.
    var notes = container.querySelectorAll('.sn-note');
    notes.forEach(function (note) {
      if (note.classList.contains('sn-note-moved-inline')) return;
      var ph = ensurePlaceholder(note);
      // Find closest paragraph and move note after it.
      var p = note.closest('p');
      if (p && p.parentNode) {
        p.parentNode.insertBefore(note, p.nextSibling);
        note.classList.add('sn-note-moved-inline');
      }
    });
  }

  function restoreInlineNotes(container) {
    var notes = container.querySelectorAll('.sn-note.sn-note-moved-inline');
    notes.forEach(function (note) {
      var id = note.getAttribute('id');
      if (!id) return;
      var ph = container.querySelector('#snph-' + id);
      if (ph && ph.parentNode) {
        ph.parentNode.insertBefore(note, ph.nextSibling);
        note.classList.remove('sn-note-moved-inline');
      }
    });
  }

  function onReady() {
    var containers = document.querySelectorAll('.sn-container');
    containers.forEach(function (c) { applyResponsiveBehavior(c); });
  }

  // Attach behaviors to handle dynamic content.
  Drupal.behaviors.sidenotes = {
    attach: function attach(context) {
      var containers = context.querySelectorAll ? context.querySelectorAll('.sn-container') : [];
      containers.forEach(function (c) { applyResponsiveBehavior(c); });
    }
  };

  if (document.readyState === 'complete' || document.readyState === 'interactive') {
    onReady();
  } else {
    document.addEventListener('DOMContentLoaded', onReady);
  }
  window.addEventListener('resize', onReady);

})(Drupal, drupalSettings);
