<?php

namespace Drupal\sidenotes\Plugin\Filter;

use Drupal\Component\Utility\Html;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\filter\Annotation\Filter;
use Drupal\filter\Plugin\FilterBase;
use Drupal\filter\FilterProcessResult;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a filter to transform simple sidenote syntax into markup.
 *
 * Supported syntaxes:
 * - [sn]...[/sn]
 * - ((...))
 *
 * @Filter(
 *   id = "filter_sidenotes",
 *   title = @Translation("Sidenotes (Tufte-style)"),
 *   description = @Translation("Create margin notes using [sn]...[/sn] or double parentheses ((...)). Adds responsive CSS/JS and optional numbering."),
 *   type = Drupal\filter\Plugin\FilterInterface::TYPE_TRANSFORM_IRREVERSIBLE,
 * )
 */
class FilterSidenotes extends FilterBase implements ContainerFactoryPluginInterface {

  /**
   * Module configuration (sidenotes.settings).
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $moduleConfig;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->moduleConfig = $container->get('config.factory')->get('sidenotes.settings');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function process($text, $langcode) {
    // Early return if nothing that looks like a sidenote exists.
    if (strpos($text, '[sn]') === FALSE && strpos($text, '((') === FALSE) {
      return new FilterProcessResult($text);
    }

    // Determine numbering/labeling mode.
    $numbered = (bool) $this->moduleConfig->get('numbered');
    $mode = $this->moduleConfig->get('numbering_mode') ?: ($numbered ? 'numbers' : 'none');
    $placement = $this->moduleConfig->get('placement') ?: 'right';
    $mobile_mode = $this->moduleConfig->get('mobile_mode') ?: 'inline'; // inline|endnotes
    $breakpoint = (int) ($this->moduleConfig->get('mobile_breakpoint') ?: 768);
    $inline_pos = $this->moduleConfig->get('mobile_inline_position') ?: 'after_paragraph';
    $allow_double_paren = (bool) $this->moduleConfig->get('syntax.double_paren');
    $allow_shortcode = (bool) $this->moduleConfig->get('syntax.shortcode');
    $unnumbered_marker = (string) ($this->moduleConfig->get('unnumbered_marker') ?? '!@');
    $symbols = $this->moduleConfig->get('symbol_set') ?: ['*','†','††','§','¶','‖','Δ','◊','☞'];
    $sym_prefix = (string) ($this->moduleConfig->get('symbol_override_prefix') ?? '!');
    $sym_suffix = (string) ($this->moduleConfig->get('symbol_override_suffix') ?? '!');
    $unnumbered_symbol = (string) ($this->moduleConfig->get('unnumbered_display_symbol') ?? '✶');
    $custom_inline_symbol = (string) ($this->moduleConfig->get('custom_label_inline_symbol') ?? '✶');

    // Parse symbol code map (lines of code=symbol).
    $code_map_raw = (string) ($this->moduleConfig->get('symbol_code_map') ?? '');
    $code_map = [];
    if ($code_map_raw !== '') {
      foreach (preg_split('/\r?\n/', $code_map_raw) as $line) {
        $line = trim($line);
        if ($line === '' || strpos($line, '=') === FALSE) { continue; }
        [$k, $v] = array_map('trim', explode('=', $line, 2));
        if ($k !== '' && $v !== '') { $code_map[$k] = $v; }
      }
    }

    // Load custom labels and order by weight.
    $custom_labels = (array) ($this->moduleConfig->get('custom_labels') ?: []);
    usort($custom_labels, function ($a, $b) { return ($a['weight'] ?? 0) <=> ($b['weight'] ?? 0); });
    $custom_labels_only = array_values(array_map(function ($r) { return (string) ($r['label'] ?? ''); }, $custom_labels));

    $noteIndex = 0; // Unique id index for notes and refs.
    $seq = 0;      // Display sequence (skips unnumbered notes).

    // Prepare symbol override regex parts (longest-first to match '††' over '†').
    $symbols_sorted = $symbols;
    usort($symbols_sorted, function ($a, $b) { return strlen($b) <=> strlen($a); });
    $sym_alternation = implode('|', array_map(function ($s) { return preg_quote($s, '/'); }, $symbols_sorted));
    $margin_width = $this->moduleConfig->get('theme.margin_width') ?: '18rem';
    $gap = $this->moduleConfig->get('theme.gap') ?: '1rem';

    $replace_callback = function (array $matches) use (&$noteIndex, &$seq, $mode, $unnumbered_marker, $unnumbered_symbol, $custom_inline_symbol, $sym_prefix, $sym_suffix, $sym_alternation, $code_map, $custom_labels, $custom_labels_only) {
      $noteIndex++;
      $content = $matches[1];

      $is_unnumbered = FALSE;
      if ($unnumbered_marker !== '') {
        // Check and strip marker at the very start (ignoring leading whitespace).
        if (preg_match('/^\s*' . preg_quote($unnumbered_marker, '/') . '\s*/u', $content, $m)) {
          $is_unnumbered = TRUE;
          $content = (string) substr($content, strlen($m[0]));
        }
      }

      // Per-note symbol override:
      // 1) Code-based, e.g., !d! for dagger, using configured prefix/suffix and code map.
      // 2) Legacy symbol-based, e.g., !† or !* (uses prefix only).
      // 3) Custom-label override by exact marker match (from settings custom_labels[].override).
      $symbol_override = NULL;
      $custom_label_override = NULL;
      if (!$is_unnumbered) {
        if ($sym_prefix !== '' && $sym_suffix !== '' && !empty($code_map)) {
          // Build alternation of known codes (longest-first).
          $codes = array_keys($code_map);
          usort($codes, function ($a, $b) { return strlen($b) <=> strlen($a); });
          $code_alt = implode('|', array_map(function ($c) { return preg_quote($c, '/'); }, $codes));
          if ($code_alt !== '') {
            if (preg_match('/^\s*' . preg_quote($sym_prefix, '/') . '(' . $code_alt . ')' . preg_quote($sym_suffix, '/') . '\s*/u', $content, $mCode)) {
              $symbol_override = $code_map[$mCode[1]] ?? NULL;
              $content = (string) substr($content, strlen($mCode[0]));
            }
          }
        }
        if ($symbol_override === NULL && $sym_prefix !== '' && $sym_alternation !== '') {
          if (preg_match('/^\s*' . preg_quote($sym_prefix, '/') . '(' . $sym_alternation . ')\s*/u', $content, $m2)) {
            $symbol_override = $m2[1];
            $content = (string) substr($content, strlen($m2[0]));
          }
        }
        // Custom label override: check for any configured override marker at start.
        if ($custom_label_override === NULL && !empty($custom_labels)) {
          foreach ($custom_labels as $cl) {
            $marker = trim((string) ($cl['override'] ?? ''));
            if ($marker === '') { continue; }
            if (preg_match('/^\s*' . preg_quote($marker, '/') . '\s*/u', $content, $m3)) {
              $custom_label_override = (string) ($cl['label'] ?? '');
              $content = (string) substr($content, strlen($m3[0]));
              break;
            }
          }
        }
      }

      // Compute label and where to show it (in ref and/or in note).
      // Do not escape here; rely on other filters in the format to sanitize.
      $label = '';
      $show_label_in_ref = FALSE;
      $show_label_in_note = FALSE;
      if ($is_unnumbered) {
        // Show only in reference, not in the note.
        $label = $unnumbered_symbol;
        $show_label_in_ref = TRUE;
        $show_label_in_note = FALSE;
      }
      else {
        if ($symbol_override !== NULL) {
          $label = $symbol_override;
          $show_label_in_ref = TRUE;
          $show_label_in_note = TRUE;
          // Only advance sequence in Symbols mode; otherwise treat as unnumbered for numbering purposes.
          if ($mode === 'symbols') { $seq++; }
        }
        else if ($custom_label_override !== NULL) {
          // Use the chosen custom label; increment only in custom mode.
          $label = $custom_label_override;
          $show_label_in_ref = TRUE; // ref will show custom inline symbol, not label text, handled below.
          $show_label_in_note = TRUE;
          if ($mode === 'custom') { $seq++; }
        }
        else if ($mode === 'numbers') {
          $seq++;
          $label = (string) $seq;
          $show_label_in_ref = TRUE;
          $show_label_in_note = TRUE;
        }
        else if ($mode === 'symbols') {
          $seq++;
          // We'll choose symbol later using the original list to preserve order in UI.
        }
        else if ($mode === 'custom') {
          // Custom labels use sequence-based label from configured list.
          if (!empty($custom_labels_only)) {
            $seq++;
            $idx = ($seq - 1) % count($custom_labels_only);
            $label = $custom_labels_only[$idx];
            $show_label_in_ref = TRUE; // But ref uses a generic/custom symbol instead of label text.
            $show_label_in_note = TRUE;
          }
        }
      }

      // If symbol mode and not overridden, compute label now using original list.
      if (!$is_unnumbered && $mode === 'symbols' && $symbol_override === NULL) {
        // Use the unsorted symbol list from config for display sequence.
        $symbols_display = $this->moduleConfig->get('symbol_set') ?: ['*','†','††','§','¶','‖'];
        $count = count($symbols_display);
        if ($count > 0) {
          $idx = ($seq - 1) % $count;
          $label = $symbols_display[$idx];
          $show_label_in_ref = TRUE;
          $show_label_in_note = TRUE;
        }
      }

      // Reference label selection: use custom inline symbol in custom mode (or when custom override used later), else label itself.
      $ref_label = '✶';
      if ($is_unnumbered) {
        $ref_label = Html::escape($label);
      } else if ($mode === 'custom' || (!empty($custom_labels_only) && $show_label_in_note && $label !== '' && in_array($label, $custom_labels_only, TRUE))) {
        $ref_label = Html::escape($custom_inline_symbol);
      } else if ($show_label_in_ref && $label !== '') {
        $ref_label = Html::escape($label);
      }

      $ref = '<sup class="sn-ref" id="snref' . $noteIndex . '"><a class="sn-ref-link" href="#sn' . $noteIndex . '" aria-label="' . (($show_label_in_ref || $show_label_in_note) && $label !== '' ? 'Sidenote ' . Html::escape(strip_tags($label)) : 'Sidenote') . '">' . $ref_label . '</a></sup>';
      $note = '<span class="sn-note" id="sn' . $noteIndex . '" data-sn-index="' . $noteIndex . '" role="note">'
        . ($show_label_in_note && $label !== '' ? '<span class="sn-note-number" aria-hidden="true">' . $label . '</span>' : '')
        . '<span class="sn-note-content">' . $content . '</span>'
        . '</span>';

      // Place the note immediately after the reference in-flow; CSS/JS will
      // float/move it to the margin or endnotes as needed.
      return $ref . $note;
    };

    $processed = $text;

    if ($allow_shortcode) {
      $processed = preg_replace_callback('/\[sn\](.*?)\[\/sn\]/s', $replace_callback, $processed);
    }

    if ($allow_double_paren) {
      // Avoid matching URLs like http://example.com/path(foo)
      // Only match when two consecutive opening parentheses are present and a matching pair of closing ones.
      $processed = preg_replace_callback('/\(\((.*?)\)\)/s', $replace_callback, $processed);
    }

    // Wrap in a container to provide layout context in CSS/JS.
    $classes = [
      'sn-container',
      $placement === 'left' ? 'sn-margin-left' : 'sn-margin-right',
      $mobile_mode === 'endnotes' ? 'sn-mobile-endnotes' : 'sn-mobile-inline',
      $mode === 'numbers' ? 'sn-numbered' : ($mode === 'symbols' ? 'sn-symbols' : 'sn-unnumbered'),
    ];

    $wrapper_attributes = [
      'class' => implode(' ', $classes),
      'data-sn-numbered' => $numbered ? '1' : '0',
      'data-sn-placement' => Html::escape($placement),
      'data-sn-mobile' => Html::escape($mobile_mode),
      'data-sn-inline-pos' => Html::escape($inline_pos),
      'data-sn-breakpoint' => (string) $breakpoint,
      'data-sn-endnotes-title' => $this->moduleConfig->get('endnotes_title') ?: 'Notes',
      'style' => '--sn-margin-width: ' . Html::escape($margin_width) . '; --sn-gap: ' . Html::escape($gap) . ';',
    ];

    $attr_html = '';
    foreach ($wrapper_attributes as $k => $v) {
      $attr_html .= ' ' . $k . '="' . Html::escape($v) . '"';
    }

    $output = '<div' . $attr_html . '>'
      . '<div class="sn-content">' . $processed . '</div>'
      . '</div>';

    $result = new FilterProcessResult($output);
    $result->setAttachments([
      'library' => ['sidenotes/sidenotes'],
      'drupalSettings' => [
        'sidenotes' => [
          'breakpoint' => $breakpoint,
        ],
      ],
    ]);

    return $result;
  }

}
