<?php

namespace Drupal\sidenotes\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

class SidenotesSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['sidenotes.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'sidenotes_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('sidenotes.settings');

    $form['placement'] = [
      '#type' => 'radios',
      '#title' => $this->t('Default margin placement'),
      '#options' => [
        'right' => $this->t('Right margin'),
        'left' => $this->t('Left margin'),
      ],
      '#default_value' => $config->get('placement') ?: 'right',
      '#description' => $this->t('Which side of the content the margin notes align to on wide screens.'),
    ];

    $current_mode = $config->get('numbering_mode');
    if (!$current_mode) {
      $current_mode = $config->get('numbered') ? 'numbers' : 'none';
    }

    $form['numbering_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Sidenote label style'),
      '#options' => [
        'numbers' => $this->t('Numbers (1, 2, 3…)'),
        'symbols' => $this->t('Symbols (*, †, ††, §, ¶, ‖)'),
        'custom' => $this->t('Custom (admin-defined labels)'),
        'none' => $this->t('None (generic mark)'),
      ],
      '#default_value' => $current_mode,
      '#description' => $this->t('Choose how sidenotes are labeled. “None” shows a generic reference mark.'),
    ];

    $form['unnumbered_marker'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Marker for unnumbered note'),
      '#default_value' => $config->get('unnumbered_marker') ?: '!@',
      '#size' => 10,
      '#maxlength' => 32,
      '#description' => $this->t('Prefix this marker at the start of a note to exclude it from numbering, e.g., [sn]!@This note is unnumbered[/sn].'),
    ];

    $form['unnumbered_display_symbol'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Symbol shown for unnumbered override'),
      '#default_value' => $config->get('unnumbered_display_symbol') ?: '✶',
      '#size' => 5,
      '#maxlength' => 8,
      '#description' => $this->t('This symbol appears by the reference and in the note when a note is marked unnumbered.'),
    ];

    $form['mobile'] = [
      '#type' => 'details',
      '#title' => $this->t('Mobile behavior'),
      '#open' => TRUE,
    ];

    $form['mobile']['mobile_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('On small screens'),
      '#options' => [
        'inline' => $this->t('Render inline after the reference'),
        'endnotes' => $this->t('Move to an endnotes section'),
      ],
      '#default_value' => $config->get('mobile_mode') ?: 'inline',
    ];

    $form['mobile']['mobile_inline_position'] = [
      '#type' => 'radios',
      '#title' => $this->t('Inline note placement (mobile)'),
      '#options' => [
        'after_paragraph' => $this->t('Place after the paragraph'),
        'after_reference' => $this->t('Place immediately after the reference'),
      ],
      '#default_value' => $config->get('mobile_inline_position') ?: 'after_paragraph',
      '#states' => [
        'visible' => [
          ':input[name="mobile_mode"]' => ['value' => 'inline'],
        ],
      ],
      '#description' => $this->t('Controls where inline notes appear on mobile when inline mode is enabled.'),
    ];

    $form['mobile']['mobile_breakpoint'] = [
      '#type' => 'number',
      '#title' => $this->t('Breakpoint (px)'),
      '#default_value' => (int) ($config->get('mobile_breakpoint') ?: 768),
      '#min' => 320,
      '#step' => 1,
      '#description' => $this->t('Viewport width below which mobile behavior applies.'),
    ];

    $form['syntax'] = [
      '#type' => 'details',
      '#title' => $this->t('Input syntax options'),
      '#open' => TRUE,
    ];

    $form['syntax']['syntax_shortcode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable [sn]...[/sn] shortcode'),
      '#default_value' => (bool) $config->get('syntax.shortcode'),
    ];

    $form['syntax']['syntax_double_paren'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable double parentheses syntax: ((...))'),
      '#default_value' => (bool) $config->get('syntax.double_paren'),
    ];

    $form['symbols'] = [
      '#type' => 'details',
      '#title' => $this->t('Symbol options'),
      '#open' => FALSE,
      '#description' => $this->t('These overrides are always available. Outside of Symbols mode, per-note symbol overrides do not increment numbering (they behave like unnumbered for numbering purposes).'),
    ];

    $form['symbols']['symbol_set'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Symbol sequence (comma-separated)'),
      '#default_value' => implode(', ', (array) ($config->get('symbol_set') ?: ['*','†','††','§','¶','‖','Δ','◊','☞'])),
      '#description' => $this->t('Used in order and cycled on overflow.'),
    ];

    $form['symbols']['symbol_override_prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Prefix for per-note symbol override'),
      '#default_value' => $config->get('symbol_override_prefix') ?: '!',
      '#size' => 10,
      '#maxlength' => 32,
      '#description' => $this->t('Used with the suffix to wrap a code, e.g., !d! for dagger.'),
    ];

    $form['symbols']['symbol_override_suffix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Suffix for per-note symbol override'),
      '#default_value' => $config->get('symbol_override_suffix') ?: '!',
      '#size' => 10,
      '#maxlength' => 32,
      '#description' => $this->t('Used with the prefix to form a code, e.g., !d! for dagger.'),
    ];

    // Build symbol code map default reordered to match symbol_set order.
    $symbol_set_array = (array) ($config->get('symbol_set') ?: ['*','†','††','§','¶','‖','Δ','◊','☞']);
    $existing_map = (string) ($config->get('symbol_code_map') ?: '');
    $map_pairs = [];
    if ($existing_map !== '') {
      foreach (preg_split('/\r?\n/', $existing_map) as $line) {
        $line = trim($line);
        if ($line === '' || strpos($line, '=') === FALSE) { continue; }
        [$k, $v] = array_map('trim', explode('=', $line, 2));
        if ($k !== '' && $v !== '') { $map_pairs[$k] = $v; }
      }
    } else {
      // Provide defaults aligned with symbol_set order when no map is saved.
      $defaults = [
        '*' => 'ast', '†' => 'd', '††' => 'dd', '§' => 'sec', '¶' => 'para', '‖' => 'parallel', 'Δ' => 'delta', '◊' => 'lozenge', '☞' => 'hand',
      ];
      foreach ($symbol_set_array as $sym) {
        $code = $defaults[$sym] ?? '';
        if ($code) { $map_pairs[$code] = $sym; }
      }
    }
    // Reorder pairs by symbol_set order primarily, then remaining codes.
    $ordered_lines = [];
    $used_codes = [];
    foreach ($symbol_set_array as $sym) {
      foreach ($map_pairs as $code => $s) {
        if ($s === $sym && !isset($used_codes[$code])) {
          $ordered_lines[] = $code . '=' . $s;
          $used_codes[$code] = TRUE;
          break;
        }
      }
    }
    foreach ($map_pairs as $code => $s) {
      if (!isset($used_codes[$code])) { $ordered_lines[] = $code . '=' . $s; }
    }
    $form['symbols']['symbol_code_map'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Symbol code map'),
      '#default_value' => implode("\n", $ordered_lines),
      '#description' => $this->t('Map mnemonic codes to symbols, one per line in the form code=symbol. Examples: d=†, dd=††, ast=*, sec=§, para=¶, parallel=‖, delta=Δ, lozenge=◊, hand=☞'),
    ];

    // Custom labels section (always available).
    // Keep custom labels fieldset open during AJAX add/remove and when rows exist
    // or when Custom mode is selected.
    $keep_open = (bool) $form_state->get('custom_keep_open');
    $saved_labels = (array) ($config->get('custom_labels') ?: []);
    usort($saved_labels, function ($a, $b) { return ($a['weight'] ?? 0) <=> ($b['weight'] ?? 0); });
    $rows = $form_state->get('custom_labels');
    if (!is_array($rows)) {
      $rows = $saved_labels;
      $form_state->set('custom_labels', $rows);
    }
    $open_custom = $keep_open || !empty($rows) || ($current_mode === 'custom');

    $form['custom'] = [
      '#type' => 'details',
      '#title' => $this->t('Custom labels'),
      '#open' => $open_custom,
      '#description' => $this->t('Define custom sidenote labels and optional override markers. These can be selected as the label style, or used individually via overrides.'),
      '#prefix' => '<div id="sn-custom-wrapper">',
      '#suffix' => '</div>',
      '#tree' => TRUE,
    ];

    // Build rows from config or form state.
    // Rows are prepared above.

    $form['custom']['table'] = [
      '#type' => 'table',
      '#header' => [
        '',
        $this->t('Label (HTML allowed)'),
        $this->t('Override marker'),
        $this->t('Weight'),
        $this->t('Operations'),
      ],
      '#empty' => $this->t('No custom labels added yet.'),
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'custom-label-weight',
        ],
      ],
      '#tree' => TRUE,
    ];

    foreach ($rows as $delta => $item) {
      $form['custom']['table'][$delta]['#attributes']['class'][] = 'draggable';
      // First cell reserved for the tabledrag handle.
      $form['custom']['table'][$delta]['_drag'] = [
        '#markup' => '',
      ];
      $form['custom']['table'][$delta]['label'] = [
        '#type' => 'textfield',
        '#default_value' => $item['label'] ?? '',
      ];
      $form['custom']['table'][$delta]['override'] = [
        '#type' => 'textfield',
        '#default_value' => $item['override'] ?? '',
      ];
      $form['custom']['table'][$delta]['weight'] = [
        '#type' => 'weight',
        '#default_value' => $item['weight'] ?? 0,
        '#attributes' => ['class' => ['custom-label-weight']],
      ];
      $form['custom']['table'][$delta]['operations'] = [
        '#type' => 'container',
      ];
      $form['custom']['table'][$delta]['operations']['remove'] = [
        '#type' => 'submit',
        '#value' => $this->t('Remove'),
        '#name' => 'custom_remove_' . $delta,
        '#submit' => [[get_class($this), 'removeRowSubmit']],
        '#limit_validation_errors' => [],
        '#ajax' => [
          'callback' => '::ajaxRebuild',
          'wrapper' => 'sn-custom-wrapper',
        ],
        '#attributes' => ['class' => ['button', 'button--danger', 'button--small']],
      ];
    }

    $form['custom']['custom_label_inline_symbol'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Inline symbol when using custom labels'),
      '#default_value' => $config->get('custom_label_inline_symbol') ?: '✶',
      '#size' => 5,
      '#maxlength' => 8,
      '#description' => $this->t('The symbol shown in the text reference when a custom label is used or when the label style is Custom.'),
    ];

    $form['custom']['add'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add custom label'),
      '#submit' => [[get_class($this), 'addCustomLabelSubmit']],
      '#limit_validation_errors' => [],
      '#ajax' => [
        'callback' => '::ajaxRebuild',
        'wrapper' => 'sn-custom-wrapper',
      ],
    ];

    // Per-row remove buttons (above) replace batch removal.

    $form['theme'] = [
      '#type' => 'details',
      '#title' => $this->t('Theme & spacing'),
      '#open' => FALSE,
    ];

    $form['theme']['margin_width'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Margin width (CSS value)'),
      '#default_value' => $config->get('theme.margin_width') ?: '18rem',
      '#description' => $this->t('Any valid CSS length, e.g., 16rem, 240px. Applied via CSS variable --sn-margin-width.'),
    ];

    $form['theme']['gap'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Gap between text and margin (CSS value)'),
      '#default_value' => $config->get('theme.gap') ?: '1rem',
      '#description' => $this->t('Applied via CSS variable --sn-gap.'),
    ];

    $form['endnotes_title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Endnotes section title'),
      '#default_value' => $config->get('endnotes_title') ?: 'Notes',
      '#description' => $this->t('Heading text for the mobile endnotes section.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);
    // Collect custom labels from table.
    $custom_rows = (array) ($form_state->getValue(['custom','table']) ?? []);
    if (empty($custom_rows)) {
      // Fallback to state (e.g., when details element state prevents values).
      $custom_rows = (array) ($form_state->get('custom_labels') ?? []);
    }
    $custom_labels = [];
    foreach ($custom_rows as $row) {
      if (!empty($row['remove'])) { continue; }
      if (($row['label'] ?? '') === '' && ($row['override'] ?? '') === '') { continue; }
      $custom_labels[] = [
        'label' => (string) $row['label'],
        'override' => (string) $row['override'],
        'weight' => (int) ($row['weight'] ?? 0),
      ];
    }
    $this->configFactory()->getEditable('sidenotes.settings')
      ->set('placement', $form_state->getValue('placement'))
      ->set('numbering_mode', $form_state->getValue('numbering_mode'))
      ->set('numbered', $form_state->getValue('numbering_mode') === 'numbers')
      ->set('unnumbered_marker', $form_state->getValue('unnumbered_marker'))
      ->set('unnumbered_display_symbol', $form_state->getValue('unnumbered_display_symbol'))
      ->set('mobile_mode', $form_state->getValue(['mobile', 'mobile_mode']) ?? $form_state->getValue('mobile_mode'))
      ->set('mobile_inline_position', $form_state->getValue(['mobile', 'mobile_inline_position']) ?? $form_state->getValue('mobile_inline_position'))
      ->set('mobile_breakpoint', (int) ($form_state->getValue(['mobile', 'mobile_breakpoint']) ?? $form_state->getValue('mobile_breakpoint')))
      ->set('symbol_set', array_values(array_filter(array_map('trim', explode(',', (string) ($form_state->getValue(['symbols', 'symbol_set']) ?? $form_state->getValue('symbol_set') ?? ''))))))
      ->set('symbol_override_prefix', $form_state->getValue(['symbols', 'symbol_override_prefix']) ?? $form_state->getValue('symbol_override_prefix'))
      ->set('symbol_override_suffix', $form_state->getValue(['symbols', 'symbol_override_suffix']) ?? $form_state->getValue('symbol_override_suffix'))
      ->set('symbol_code_map', $form_state->getValue(['symbols', 'symbol_code_map']) ?? $form_state->getValue('symbol_code_map'))
      ->set('custom_labels', $custom_labels)
      ->set('custom_label_inline_symbol', $form_state->getValue(['custom','custom_label_inline_symbol']) ?? $form_state->getValue('custom_label_inline_symbol'))
      ->set('syntax.shortcode', (bool) ($form_state->getValue(['syntax', 'syntax_shortcode']) ?? $form_state->getValue('syntax_shortcode')))
      ->set('syntax.double_paren', (bool) ($form_state->getValue(['syntax', 'syntax_double_paren']) ?? $form_state->getValue('syntax_double_paren')))
      ->set('theme.margin_width', $form_state->getValue(['theme', 'margin_width']) ?? $form_state->getValue('margin_width'))
      ->set('theme.gap', $form_state->getValue(['theme', 'gap']) ?? $form_state->getValue('gap'))
      ->set('endnotes_title', $form_state->getValue('endnotes_title'))
      ->save();
  }

  /**
   * AJAX callback to rebuild the custom labels section.
   */
  public function ajaxRebuild(array &$form, FormStateInterface $form_state) {
    return $form['custom'];
  }

  /**
   * Submit handler to add a custom label row.
   */
  public static function addCustomLabelSubmit(array &$form, FormStateInterface $form_state) {
    // Sync current posted rows into state first.
    $posted = (array) ($form_state->getValue(['custom','table']) ?? []);
    if (!empty($posted)) {
      $synced = [];
      foreach ($posted as $row) {
        $synced[] = [
          'label' => (string) ($row['label'] ?? ''),
          'override' => (string) ($row['override'] ?? ''),
          'weight' => (int) ($row['weight'] ?? 0),
        ];
      }
      $form_state->set('custom_labels', $synced);
    }
    $rows = (array) $form_state->get('custom_labels') ?: [];
    $rows[] = ['label' => '', 'override' => '', 'weight' => count($rows)];
    $form_state->set('custom_labels', $rows);
    $form_state->set('custom_keep_open', TRUE);
    $form_state->setRebuild(TRUE);
  }

  /**
   * Submit handler to remove selected custom labels.
   */
  public static function removeCustomLabelsSubmit(array &$form, FormStateInterface $form_state) {
    // No-op: replaced by per-row remove.
  }

  /**
   * Submit handler to remove a specific row via AJAX.
   */
  public static function removeRowSubmit(array &$form, FormStateInterface $form_state) {
    // Determine which button was clicked.
    $trigger = $form_state->getTriggeringElement();
    $name = is_array($trigger) && isset($trigger['#name']) ? $trigger['#name'] : '';
    $remove_delta = NULL;
    if (is_string($name) && preg_match('/^custom_remove_(\d+)$/', $name, $m)) {
      $remove_delta = (int) $m[1];
    }
    // Sync posted rows into state, then remove the target delta.
    $posted = (array) ($form_state->getValue(['custom','table']) ?? []);
    $kept = [];
    foreach ($posted as $delta => $row) {
      if ($remove_delta !== NULL && (int) $delta === $remove_delta) { continue; }
      $kept[] = [
        'label' => (string) ($row['label'] ?? ''),
        'override' => (string) ($row['override'] ?? ''),
        'weight' => (int) ($row['weight'] ?? 0),
      ];
    }
    // Reindex weights.
    foreach ($kept as $i => &$row) { $row['weight'] = $i; }
    unset($row);
    $form_state->set('custom_labels', $kept);
    $form_state->set('custom_keep_open', TRUE);
    $form_state->setRebuild(TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
    if ($form_state->getValue('numbering_mode') === 'custom') {
      $rows = (array) ($form_state->getValue(['custom','table']) ?? []);
      $has = FALSE;
      foreach ($rows as $row) {
        if (!empty($row['label'])) { $has = TRUE; break; }
      }
      if (!$has) {
        $form_state->setErrorByName('numbering_mode', $this->t('Custom labels chosen as label style, but no custom labels are defined. Add at least one custom label or choose a different style.'));
      }
    }
  }

}
