<?php

namespace Drupal\simple_crop\Plugin\Field\FieldType;

use Drupal\image\Plugin\Field\FieldType\ImageItem;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\TypedData\DataDefinition;

/**
 * Plugin implementation of the 'simple_crop' field type.
 *
 * @FieldType(
 *   id = "simple_crop",
 *   label = @Translation("Simple Crop Image"),
 *   description = @Translation("Image field with simple crop support."),
 *   default_widget = "simple_crop_widget",
 *   default_formatter = "image",
 *   target_type = "file",
 *   list_class = "\Drupal\Core\Field\EntityReferenceFieldItemList"
 * )
 */
class SimpleCropItem extends ImageItem {

  public function preSave() {
    parent::preSave();

    if (empty($this->target_id)) {
      return;
    }

    /** @var \Drupal\file\FileInterface|null $file */
    $file = \Drupal::entityTypeManager()->getStorage('file')->load($this->target_id);
    if (!$file) {
      return;
    }

    // Final output dims from field settings (your defaults are fine).
    $outW = (int) ($this->getSetting('aspect_x') ?: 390);
    $outH = (int) ($this->getSetting('aspect_y') ?: 280);

    $coords = [
      'crop_x'      => (int) $this->crop_x,
      'crop_y'      => (int) $this->crop_y,
      'crop_width'  => (int) $this->crop_width,
      'crop_height' => (int) $this->crop_height,
    ];
    if ($coords['crop_width'] <= 0 || $coords['crop_height'] <= 0) {
      return; // Nothing to crop.
    }

    $context = [
      'entity_type' => $this->getEntity()->getEntityTypeId(),
      'entity_id'   => $this->getEntity()->id(),
      'field'       => $this->getFieldDefinition()->getName(),
    ];

    /** @var \Drupal\simple_crop\SimpleCropProcessor $proc */
    $proc = \Drupal::service('simple_crop.processor');
    $dest = $proc->process(
      $file,
      $coords,
      ['width' => $outW, 'height' => $outH],
      $context
    );

    if ($dest) {
      // Reflect final dims for formatters.
      $this->width = $outW;
      $this->height = $outH;

      // Optional helpers if you added these columns.
      if (property_exists($this, 'processed')) {
        $this->processed = 1;
      }
      if (property_exists($this, 'derivative_uri')) {
        $this->derivative_uri = $dest;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    $properties = parent::propertyDefinitions($field_definition);

    $properties['crop_x'] = DataDefinition::create('integer')->setLabel(t('Crop X'));
    $properties['crop_y'] = DataDefinition::create('integer')->setLabel(t('Crop Y'));
    $properties['crop_width'] = DataDefinition::create('integer')->setLabel(t('Crop Width'));
    $properties['crop_height'] = DataDefinition::create('integer')->setLabel(t('Crop Height'));
    $properties['processed'] = DataDefinition::create('boolean')->setLabel(t('Processed'));
    $properties['derivative_uri'] = DataDefinition::create('string')->setLabel(t('Cropped image URI'));

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    $schema = parent::schema($field_definition);
    $schema['columns']['crop_x'] = ['type' => 'int', 'unsigned' => TRUE, 'default' => 0];
    $schema['columns']['crop_y'] = ['type' => 'int', 'unsigned' => TRUE, 'default' => 0];
    $schema['columns']['crop_width'] = ['type' => 'int', 'unsigned' => TRUE, 'default' => 0];
    $schema['columns']['crop_height'] = ['type' => 'int', 'unsigned' => TRUE, 'default' => 0];
    $schema['columns']['processed'] = ['type' => 'int', 'size' => 'tiny', 'unsigned' => TRUE, 'default' => 0];
    $schema['columns']['derivative_uri'] = ['type' => 'varchar', 'length' => 255, 'not null' => FALSE];
    return $schema;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings() {
    return [
      'aspect_x' => 390,
      'aspect_y' => 280,
    ] + parent::defaultFieldSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state): array {
    $element = [];

    $aspect_x = (int) ($this->getSetting('aspect_x') ?? 390);
    $aspect_y = (int) ($this->getSetting('aspect_y') ?? 280);

    $element['aspect_x'] = [
      '#type' => 'number',
      '#title' => $this->t('Aspect X'),
      '#default_value' => $aspect_x,
      '#value' => $aspect_x,
      '#min' => 1,
      '#required' => TRUE,
      '#description' => $this->t('Horizontal ratio component (e.g. 390).'),
    ];

    $element['aspect_y'] = [
      '#type' => 'number',
      '#title' => $this->t('Aspect Y'),
      '#default_value' => $aspect_y,
      '#value' => $aspect_y,
      '#min' => 1,
      '#required' => TRUE,
      '#description' => $this->t('Vertical ratio component (e.g. 280).'),
    ];

    return $element;
  }

  public function settingsSummary() {
    $summary = [];
    $summary[] = $this->t('Aspect ratio: @x:@y', [
      '@x' => (int) $this->getSetting('aspect_x'),
      '@y' => (int) $this->getSetting('aspect_y'),
    ]);
    return $summary;
  }

}

