<?php

namespace Drupal\simple_crop;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Image\ImageFactory;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\file\FileInterface;

/**
 * Handles cropping and resizing images.
 */
class SimpleCropProcessor {

  public function __construct(
    private ImageFactory $imageFactory,
    private FileSystemInterface $fileSystem,
    private LoggerChannelFactoryInterface $loggerFactory,
  ) {}

  /**
   * Crop & resize once. Updates the file entity to point at the derivative.
   *
   * @param \Drupal\file\FileInterface $file
   * @param array $crop_data    Keys: x|crop_x, y|crop_y, width|crop_width, height|crop_height (px in original space)
   * @param array $output_size  Keys: width, height (final px)
   * @param array $context      Optional: ['entity_type' => ..., 'entity_id' => ..., 'field' => ...]
   *
   * @return string|null Destination URI (or NULL on no-op/failure).
   */
  public function process(FileInterface $file, array $crop_data, array $output_size, array $context = []): ?string {
    $logger = $this->loggerFactory->get('simple_crop');

    // Normalize keys (accept either your old keys or the new crop_* ones).
    $x = (int) ($crop_data['x'] ?? $crop_data['crop_x'] ?? 0);
    $y = (int) ($crop_data['y'] ?? $crop_data['crop_y'] ?? 0);
    $w = (int) ($crop_data['width'] ?? $crop_data['crop_width'] ?? 0);
    $h = (int) ($crop_data['height'] ?? $crop_data['crop_height'] ?? 0);

    $outW = (int) ($output_size['width'] ?? 0);
    $outH = (int) ($output_size['height'] ?? 0);

    if ($w <= 0 || $h <= 0 || $outW <= 0 || $outH <= 0) {
      return NULL; // nothing to do
    }

    $srcUri = $file->getFileUri();
    $ext = strtolower(pathinfo($srcUri, PATHINFO_EXTENSION));
    if (in_array($ext, ['svg', 'svgz'], TRUE)) {
      // Skip vector files; they shouldn't be cropped here.
      return NULL;
    }

    // Build an idempotency signature: same inputs => same filename.
    $sig = sha1($file->id() . "|$x|$y|$w|$h|$outW|$outH|v1");

    // If already processed to this signature, bail.
    if (str_contains($srcUri, '/simple_crop/processed/') && str_contains($srcUri, $sig)) {
      return $srcUri;
    }

    // Destination layout (stable by entity/field if provided).
    $et  = $context['entity_type'] ?? 'file';
    $eid = $context['entity_id'] ?? $file->id();
    $fld = $context['field']      ?? 'image';

    $destDir = "public://simple_crop/processed/$et/$eid/$fld";
    $this->fileSystem->prepareDirectory(
      $destDir,
      FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS
    );

    $destUri = $destDir . '/' . $file->id() . '-' . $sig . '.' . $ext;

    // Move original to destination name so we operate in-place from here on.
    // (Optional: copy the original to an "originals" dir first if you want to preserve it.)
    try {
      $this->fileSystem->move($srcUri, $destUri, FileSystemInterface::EXISTS_REPLACE);
    }
    catch (\Throwable $e) {
      $logger->error('Move failed @src → @dst: @msg', ['@src' => $srcUri, '@dst' => $destUri, '@msg' => $e->getMessage()]);
      return NULL;
    }

    // Use Drupal image toolkit (GD or Imagick) instead of raw GD calls.
    $image = $this->imageFactory->get($destUri);
    if (!$image->isValid()) {
      $logger->error('Image invalid at @uri.', ['@uri' => $destUri]);
      return NULL;
    }

    // Crop then resize.
    $image->crop($x, $y, $w, $h);
    $image->resize($outW, $outH);
    $image->save();

    // Update file entity to the new URI & size.
    $realpath = $this->fileSystem->realpath($destUri);
    clearstatcache(TRUE, $realpath);
    if (is_file($realpath)) {
      $file->setFileUri($destUri);
      $file->setSize(filesize($realpath));
      $file->save();
    }

    return $destUri;
  }
}

