(function ($, Drupal, drupalSettings) {
  let attachCount = 0;
  window.simpleCropConfigCache = window.simpleCropConfigCache || {};

  class SimpleCropWidget {
    constructor($wrapper, fieldKey) {
      this.$wrapper = $wrapper;
      this.fieldKey = fieldKey;
      this.config = window.simpleCropConfigCache[fieldKey] || null;
    }

    init() {
      console.log('init called');
      if (!this.$wrapper.length) return;
      if (this.$wrapper.data('simpleCropInit')) return;
      this.$wrapper.data('simpleCropInit', true);

      console.log('Call createPreviewDiv();');
      this.createPreviewDiv();

   // this.$wrapper.on('click', '.js-sc-recrop', () => {
     // this.$wrapper.attr('data-sc-processed', '0');
     // this.createPreview();
   // });


      // If image already present, build preview; else watch for it.
      if (this.hasImage()) {
        console.log('Step 4b has image.');
        this.createPreview();
      }
      else {
        console.log('Step 4c watch for image upload.');
        this.watchForImageUpload();
      }
    }

    hasImage() {
      return this.$wrapper.find('.file--image a, .image-preview img').length > 0;
    }

    createPreviewDiv() {
      if (!this.$wrapper.find('.photo-preview').length) {
        const inst = this.$wrapper.attr('data-sc-instance') || '';
        const id   = inst ? ` id="photo-preview-${inst}"` : '';
        const $el  = $(`<div${id} class="photo-preview">&nbsp;</div>`);
        this.$wrapper.append($el);
      }
    }

    // Store the observer on the wrapper so we don't stack multiples.
    observeImagePreview() {
      const target = this.$wrapper[0];
      if (!target) return;

      // Disconnect previous observer if any.
      const prev = this.$wrapper.data('simpleCropObserver');
      if (prev) {
        console.log('Disconnnect previous observer.');
        prev.disconnect();
      }

      const observer = new MutationObserver((mutations, instance) => {
        if (this.hasImage()) {
          console.log('Has image.');
          this.createPreview();
          instance.disconnect();
          this.$wrapper.removeData('simpleCropObserver');
        }
        else {
          console.log('Observer has not found image yet.');
        }
      });

      console.log('Observing...');
      observer.observe(target, { childList: true, subtree: true });
      this.$wrapper.data('simpleCropObserver', observer);
    }

    watchForImageUpload() {
      const uploadButton = this.$wrapper.find('.js-form-submit[data-drupal-selector$="upload-button"]');

      // If no button (already uploaded or replaced by AJAX), just observe.
      if (!uploadButton.length) {
        console.log('Call observe image preview..');
        this.observeImagePreview();
        return;
      }

      // Avoid stacking handlers.
      uploadButton.off('mousedown.simpleCropUpload')
               .on('mousedown.simpleCropUpload', () => {
                 console.log('Upload button mousedown. Watching for new image...');
                 this.observeImagePreview();
               });
    }

    createPreview() {
      const previewContainer = this.$wrapper.find('.photo-preview')[0];
      if (!previewContainer) return;

      const link = this.$wrapper.find('.file--image a, .file-link a').get(0);
      if (!link) return;

      const originalUrl = link.getAttribute('href');

      // Determine processed state:
      //  - If the wrapper says it's processed, honor that.
      //  - Or infer from the URL path (your server moves processed files under /simple_crop/processed/).
      const processedAttr = this.$wrapper.attr('data-sc-processed') === '1';
      const processedUrl  = /\/simple_crop\/processed\//.test(originalUrl);
      const isProcessed   = processedAttr || processedUrl;

      const candidateUrl = originalUrl.replace(/\/(\d+)(\/[^\/]+\.(?:jpg|jpeg|png|gif))/i, '/$1/orig$2');
      const testImage = new Image();
      const style = 'max-height: 1000px; max-width: 1000px; object-fit: contain;';

      const render = (src) => {
        previewContainer.innerHTML = `<img class="simple-crop-preview-img" src="${src}" style="${style}" />`;

        // Only build the overlay if we’re NOT already processed.
        if (!isProcessed) {
          this.photoFocusOverlay(previewContainer, this.config);
          previewContainer.classList.remove('sc-processed');
        }
        else {
          previewContainer.classList.add('sc-processed'); // optional styling hook
        }
      };

      testImage.onload = () => render(candidateUrl);
      testImage.onerror = () => render(originalUrl);
      testImage.src = candidateUrl;
    }

    photoFocusOverlay($el, config) {
      const previewContainer = $el;
      const image = previewContainer?.querySelector('img');

      if (!image || previewContainer.querySelector('.focus-overlay')) return;

      function syncPreviewSize() {
        const rect = image.getBoundingClientRect();
        previewContainer.style.width = rect.width + 'px';
        previewContainer.style.height = rect.height + 'px';
      }

      function updateCoordsInput(x, y, w, h, containerW, containerH) {
        const naturalW = image.naturalWidth;
        const naturalH = image.naturalHeight;
        const scaleX = naturalW / containerW;
        const scaleY = naturalH / containerH;

        const json = JSON.stringify({
          x: x * scaleX,
          y: y * scaleY,
          w: w * scaleX,
          h: h * scaleY,
        });

        // Scoped to THIS widget’s wrapper:
        const host  = previewContainer.closest('.field--type-simple-crop, .form-type--managed-file') || this?.$wrapper?.get?.(0);
        const input = host && host.querySelector('input[name$="[focus_box_coords]"]');
        if (input) input.value = json;
      }

      function enforceAspectRatio(box, container) {
        const containerRect = container.getBoundingClientRect();
        const aspectRatio = config.aspect_y / config.aspect_x;

        let maxWidth = containerRect.width;
        let maxHeight = containerRect.height;

        // Limit width to container width
        let width = Math.min(box.offsetWidth, maxWidth);
        let height = width * aspectRatio;

        // If height would overflow, scale both down proportionally
        if (height > maxHeight) {
          height = maxHeight;
          width = height / aspectRatio;
        }

        // Clamp position so box stays fully within container
        let left = parseFloat(box.style.left) || 0;
        let top = parseFloat(box.style.top) || 0;

        left = Math.min(left, containerRect.width - width);
        top = Math.min(top, containerRect.height - height);

        box.style.width = width + 'px';
        box.style.height = height + 'px';
        box.style.left = left + 'px';
        box.style.top = top + 'px';

        updateCoordsInput(left, top, width, height, containerRect.width, containerRect.height);
      }

      function initOverlay() {
        syncPreviewSize();
        previewContainer.style.position = 'relative';

        previewContainer.querySelectorAll('.focus-overlay, .focus-box').forEach(el => el.remove());

        const overlay = document.createElement('div');
        overlay.className = 'focus-overlay';
        overlay.innerHTML = '<div class="focus-marker">+</div>';
        previewContainer.appendChild(overlay);

        const focusBox = document.createElement('div');
        focusBox.className = 'focus-box';
        focusBox.innerHTML = '<div class="resize-handle"></div>';
        previewContainer.appendChild(focusBox);

        const imgRect = image.getBoundingClientRect();
        const containerWidth = imgRect.width;
        const containerHeight = imgRect.height;

        const input = document.querySelector('input[name="focus_box_coords"]');
        let boxWidth, boxHeight, left, top;
        if (input && input.value) {
          try {
            const prev = JSON.parse(input.value);
            const scaleX = containerWidth / image.naturalWidth;
            const scaleY = containerHeight / image.naturalHeight;
            boxWidth = prev.w * scaleX;
            boxHeight = prev.h * scaleY;
            left = prev.x * scaleX;
            top = prev.y * scaleY;
            if (top <= 0 && boxHeight < containerHeight) {
              top = Math.max((containerHeight - boxHeight) / 2, 1);
            }
          } catch (e) {
            boxWidth = containerWidth * 0.6;
            boxHeight = boxWidth * (config.aspect_y / config.aspect_x);
            left = (containerWidth - boxWidth) / 2;
            top = (containerHeight - boxHeight) / 2;
          }
        }
        else {
          boxWidth = containerWidth * 0.6;
          boxHeight = boxWidth * (config.aspect_y / config.aspect_x);
          left = (containerWidth - boxWidth) / 2;
          top = (containerHeight - boxHeight) / 2;
        }

        focusBox.style.cssText = `
          position: absolute;
          border: 2px dashed red;
          z-index: 11;
          overflow: hidden;
          touch-action: pan-y;
          left: ${left}px;
          top: ${top}px;
          width: ${boxWidth}px;
          height: ${boxHeight}px;
        `;

        const resizeHandle = focusBox.querySelector('.resize-handle');
        resizeHandle.style.cssText = `
          position: absolute;
          right: 0;
          bottom: 0;
          width: 20px;
          height: 20px;
          z-index: 12;
          cursor: nwse-resize;
          touch-action: none;
        `;

        let isDragging = false;
        let dragOffsetX, dragOffsetY;

        focusBox.addEventListener('mousedown', (e) => {
          if (e.target === resizeHandle) return;
          e.preventDefault();
          const rect = focusBox.getBoundingClientRect();
          isDragging = 'move';
          dragOffsetX = e.clientX - rect.left;
          dragOffsetY = e.clientY - rect.top;
        });

        window.addEventListener('mousemove', (e) => {
          if (isDragging !== 'move') return;
          e.preventDefault();
          const containerRect = previewContainer.getBoundingClientRect();
          let newLeft = e.clientX - containerRect.left - dragOffsetX;
          let newTop = e.clientY - containerRect.top - dragOffsetY;
          newLeft = Math.max(0, Math.min(newLeft, containerRect.width - focusBox.offsetWidth));
          newTop = Math.max(0, Math.min(newTop, containerRect.height - focusBox.offsetHeight));
          focusBox.style.left = newLeft + 'px';
          focusBox.style.top = newTop + 'px';
          updateCoordsInput(newLeft, newTop, focusBox.offsetWidth, focusBox.offsetHeight, containerRect.width, containerRect.height);
        });

        let resizeStartX;
        resizeHandle.addEventListener('mousedown', (e) => {
          e.preventDefault();
          e.stopPropagation();
          resizeStartX = e.clientX;
          isDragging = 'resize';
          focusBox.dataset.startWidth = focusBox.offsetWidth;
        });

        window.addEventListener('mousemove', (e) => {
          if (isDragging !== 'resize') return;
          e.preventDefault();
          const dx = e.clientX - resizeStartX;
          const newWidth = parseFloat(focusBox.dataset.startWidth) + dx;
          focusBox.style.width = newWidth + 'px';
          enforceAspectRatio(focusBox, previewContainer);
        });

        window.addEventListener('mouseup', () => {
          isDragging = false;
        });

        previewContainer.addEventListener('touchstart', (e) => {
          // Default: allow vertical page scroll unless we're dragging/pinching.
          if (e.touches.length === 2) {
            // Pinch-to-resize: disable scroll during this gesture.
            focusBox.style.touchAction = 'none';
            const dx = e.touches[1].clientX - e.touches[0].clientX;
            const dy = e.touches[1].clientY - e.touches[0].clientY;
            previewContainer.dataset.pinchStartDistance = Math.hypot(dx, dy);
            previewContainer.dataset.startWidth = focusBox.offsetWidth;
          }
          else if (e.touches.length === 1) {
            // One-finger drag inside the box: disable scroll during the drag.
            focusBox.style.touchAction = 'none';
            const touch = e.touches[0];
            const rect = focusBox.getBoundingClientRect();
            const containerRect = previewContainer.getBoundingClientRect();
            focusBox.dataset.dragging = 'true';
            focusBox.dataset.offsetX = touch.clientX - rect.left;
            focusBox.dataset.offsetY = touch.clientY - rect.top;
            focusBox.dataset.containerLeft = containerRect.left;
            focusBox.dataset.containerTop = containerRect.top;
          }
        }, { passive: false });

        previewContainer.addEventListener('touchmove', (e) => {
          const containerRect = previewContainer.getBoundingClientRect();
          if (e.touches.length === 2) {
            e.preventDefault();
            const dx = e.touches[1].clientX - e.touches[0].clientX;
            const dy = e.touches[1].clientY - e.touches[0].clientY;
            const distance = Math.hypot(dx, dy);
            const scale = distance / parseFloat(previewContainer.dataset.pinchStartDistance);
            let newWidth = parseFloat(previewContainer.dataset.startWidth) * scale;
            focusBox.style.width = newWidth + 'px';
            enforceAspectRatio(focusBox, previewContainer);
          }
          else if (e.touches.length === 1 && focusBox.dataset.dragging === 'true') {
            e.preventDefault();
            const touch = e.touches[0];
            const offsetX = parseFloat(focusBox.dataset.offsetX);
            const offsetY = parseFloat(focusBox.dataset.offsetY);
            const containerLeft = parseFloat(focusBox.dataset.containerLeft);
            const containerTop = parseFloat(focusBox.dataset.containerTop);
            let newLeft = touch.clientX - containerLeft - offsetX;
            let newTop = touch.clientY - containerTop - offsetY;
            newLeft = Math.max(0, Math.min(newLeft, containerRect.width - focusBox.offsetWidth));
            newTop = Math.max(0, Math.min(newTop, containerRect.height - focusBox.offsetHeight));
            focusBox.style.left = newLeft + 'px';
            focusBox.style.top = newTop + 'px';
            updateCoordsInput(newLeft, newTop, focusBox.offsetWidth, focusBox.offsetHeight, containerRect.width, containerRect.height);
          }
        }, { passive: false });

        previewContainer.addEventListener('touchend', () => {
          focusBox.dataset.dragging = 'false';
          // Re-enable the page scroll when the gesture ends.
          focusBox.style.touchAction = 'pan-y';
        });

        enforceAspectRatio(focusBox, previewContainer);
        // Commit the initial coords so Save-without-drag still has data.
        const rect = focusBox.getBoundingClientRect();
        const cont = previewContainer.getBoundingClientRect();
        updateCoordsInput(
          parseFloat(focusBox.style.left) || 0,
          parseFloat(focusBox.style.top) || 0,
          rect.width,
          rect.height,
          cont.width,
          cont.height
        );

        window.addEventListener('resize', syncPreviewSize);
      }

      if (image.complete) {
        initOverlay();
      }
      else {
        image.addEventListener('load', initOverlay);
      }
    }
  }

  // Extract field key (e.g. "field_testjojo") from a wrapper.
  function getFieldKeyFromWrapper($wrapper) {
    // target the upload input; it has a stable data-drupal-selector
    var sel = $wrapper.find('.form-managed-file__main input.form-file').attr('data-drupal-selector') || '';
    // Accept cases like: edit-field-testjojo-0-upload, edit-field-testjojo-0-upload-<random>
    var m = sel.match(/^edit-(field-[\w-]+?)-upload(?:-[\w-]+)?$/);
    if (sel == '') {
      // Image already loaded, likely has a remove button.
      sel = $wrapper.find('.form-managed-file__main span.file').attr('data-drupal-selector') || '';
      m = sel.match(/^edit-(field-[\w-]+?)-file(?:-[\w-]+)?$/);
    }
    if (!m) return null;
    const raw = m[1].replace(/-\d+$/, ''); // drop trailing -0
    return raw.replace(/-/g, '_');         // normalize to drupalSettings key
  }

  // Load config for a field once and cache globally by field key.
  function ensureConfigFor($wrapper) {
    const key = getFieldKeyFromWrapper($wrapper);
    if (!key) return null;
    if (!window.simpleCropConfigCache[key]) {
      window.simpleCropConfigCache[key] = drupalSettings.simpleCrop?.[key] || null;
    }
    return key;
  }

  Drupal.behaviors.simpleCropWidget = {
    attach(context) {
      attachCount++;
      // Iterate ALL wrappers in this context, not just .first().
      once('simple-crop-wrap', '.field--type-simple-crop.form-wrapper', context).forEach((el) => {
        const $wrapper = $(el);

        const fieldKey = ensureConfigFor($wrapper);
        if (!fieldKey || !window.simpleCropConfigCache[fieldKey]) return;

        const widget = new SimpleCropWidget($wrapper, fieldKey);
        widget.init();
      });
      console.log('Attach fire ' + attachCount);
    }
  };
})(jQuery, Drupal, drupalSettings);

