<?php

namespace Drupal\simple_crop\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\image\Plugin\Field\FieldWidget\ImageWidget;
use Drupal\file\FileInterface;

/**
 * Plugin implementation of the 'simple_crop_widget' widget.
 *
 * @FieldWidget(
 *   id = "simple_crop_widget",
 *   label = @Translation("Simple crop widget"),
 *   field_types = {
 *     "simple_crop"
 *   }
 * )
 */
class SimpleCropWidget extends ImageWidget {

  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element = parent::formElement($items, $delta, $element, $form, $form_state);
    $item = $items[$delta] ?? NULL;
    $processed = FALSE;
    $field_name = $this->fieldDefinition->getName();
    $instance_id = $field_name . '-' . $delta;

    // Hidden JSON for coords (JS writes absolute px here).
    $element['focus_box_coords'] = [
      '#type' => 'hidden',
      '#default_value' => '',
    ];

    // Prefer your field property if you kept it.
    if ($item && isset($item->processed) && (int) $item->processed === 1) {
      $processed = TRUE;
    }
    // Fallback: inspect the file URI.
    elseif ($item && !empty($item->target_id)) {
      /** @var \Drupal\file\FileInterface|null $f */
      $f = \Drupal::entityTypeManager()->getStorage('file')->load($item->target_id);
      if ($f instanceof FileInterface) {
        $processed = str_contains($f->getFileUri(), '/simple_crop/processed/');
      }
    }

    // Preview container your JS targets.
    $element['photo_preview'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'photo-preview-' . $instance_id,
        'class' => ['photo-preview'],
        'data-sc-instance' => $instance_id,
        'data-sc-processed' => $processed ? '1' : '0',
      ],
      '#weight' => 50,
    ];

    // Pass aspect settings to JS.
    $field_settings = $this->getFieldSettings();
    $element['#attached']['drupalSettings']['simpleCrop'][$field_name] = [
      'aspect_x' => $field_settings['aspect_x'] ?? 390,
      'aspect_y' => $field_settings['aspect_y'] ?? 280,
      'instance' => $instance_id,
    ];

    // Ensure AJAX upload/remove do NOT trigger full validation.
    // Keep upload/remove from triggering full validation.
    $parents = array_merge($element['#field_parents'] ?? [], [
      $this->fieldDefinition->getName(),
      $delta,
    ]);

    if (isset($element['upload_button'])) {
      // Only the file input needs to validate during upload.
      $element['upload_button']['#limit_validation_errors'] = [array_merge($parents, ['fids'])];
    }

    if (isset($element['remove_button'])) {
      // Removing also shouldn’t validate the whole node.
      $element['remove_button']['#limit_validation_errors'] = [array_merge($parents)];
    }

    $element['#attached']['library'][] = 'simple_crop/widget';
    return $element;
  }

  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    foreach ($values as &$value) {
      if (!empty($value['focus_box_coords'])) {
        $c = json_decode($value['focus_box_coords'], TRUE);
        if (is_array($c) && isset($c['x'],$c['y'],$c['w'],$c['h'])) {
          $value['crop_x'] = (int) round($c['x']);
          $value['crop_y'] = (int) round($c['y']);
          $value['crop_width'] = (int) round($c['w']);
          $value['crop_height'] = (int) round($c['h']);
        }
      }
      unset($value['focus_box_coords']);
    }

    // 2) Let core do its job: fids -> target_id, alt/title normalization, etc.
    $values = parent::massageFormValues($values, $form, $form_state);

    // 3) Merge our coords back in, per delta.
    foreach ($values as $delta => &$value) {
      if (isset($coordsByDelta[$delta])) {
        $value += $coordsByDelta[$delta];
      }
    }
    \Drupal::logger('simple_crop')->notice("WIDGET AFTER massage:\n<pre>@v</pre>", ['@v' => print_r($values, TRUE)]);
    return $values;
  }
}

