<?php

namespace Drupal\simple_crop;

use Drupal\file\FileInterface;
use Drupal\Core\File\FileSystemInterface;
use Psr\Log\LoggerInterface;

/**
 * Handles cropping and resizing images.
 */
class SimpleCropProcessor {

  protected FileSystemInterface $fileSystem;
  protected LoggerInterface $logger;

  public function __construct(FileSystemInterface $file_system, LoggerInterface $logger_factory) {
    $this->fileSystem = $file_system;
    $this->logger = $logger_factory->get('simple_crop');
  }

  /**
   * Crop and resize the given image using crop data and field settings.
   *
   * @param \Drupal\file\FileInterface $file
   *   The original uploaded image file.
   * @param array $crop_data
   *   ['x' => int, 'y' => int, 'width' => int, 'height' => int]
   * @param array $output_size
   *   ['width' => int, 'height' => int]
   *
   * @return string|null
   *   The URI to the saved derivative, or NULL on failure.
   */
  public function process(FileInterface $file, array $crop_data, array $output_size): ?string {
    $source_path = $this->fileSystem->realpath($file->getFileUri());
    if (!file_exists($source_path)) {
      $this->logger->error('Source file not found: @path', ['@path' => $source_path]);
      return NULL;
    }

    $destination_dir = 'public://simple_crop/';
    $this->fileSystem->prepareDirectory($destination_dir, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);
    $destination_uri = $destination_dir . $file->id() . '_crop.jpg';
    $destination_path = $this->fileSystem->realpath($destination_uri);

    try {
      // Use GD for now. Swap with ImageMagick if needed.
      [$x, $y, $w, $h] = [
        $crop_data['x'], $crop_data['y'],
        $crop_data['width'], $crop_data['height']
      ];

      [$out_w, $out_h] = [$output_size['width'], $output_size['height']];

      $image = imagecreatefromstring(file_get_contents($source_path));
      if (!$image) {
        throw new \Exception('Failed to load source image.');
      }

      $cropped = imagecrop($image, ['x' => $x, 'y' => $y, 'width' => $w, 'height' => $h]);
      $resized = imagecreatetruecolor($out_w, $out_h);
      imagecopyresampled($resized, $cropped, 0, 0, 0, 0, $out_w, $out_h, $w, $h);

      imagejpeg($resized, $destination_path, 90);
      imagedestroy($image);
      imagedestroy($cropped);
      imagedestroy($resized);

      return $destination_uri;
    }
    catch (\Throwable $e) {
      $this->logger->error('Image crop failed: @msg', ['@msg' => $e->getMessage()]);
      return NULL;
    }
  }

}

