/**
 * @file
 * Defines the simpleFavs Behaviour.
 */

(function ($, Drupal) {
  Drupal.behaviors.simpleFavs = {

    /**
     * Adds the current non-node page (e.g., view, term) to the favourites.
     *
     * @param {string} title
     *   The title of the current page.
     */
    addToFavsOther: function (title) {
      const path = window.location.pathname;

      Drupal.behaviors.simpleFavs.getFavsOther().then((favs) => {
        // Check if the page is already in favs.
        if (!favs.some(fav => fav.path === path)) {
          favs.push({ path: path, title: title });
          Drupal.behaviors.simpleFavs.setFavsOther(favs);
        }
      });
    },

    /**
     * Adds the current non-node page (e.g., view, term) to the favourites.
     *
     * @param {string} title
     *   The title of the current item.
     * @param {string} path
     *   The path of the current item.
     */
    addToFavsOtherFromViews: function (title, path) {
      Drupal.behaviors.simpleFavs.getFavsOther().then((favs) => {
        // Check if the page is already in favs.
        if (!favs.some(fav => fav.path === path)) {
          favs.push({ path: path, title: title });
          Drupal.behaviors.simpleFavs.setFavsOtherFromViews(favs);
        }
      });
    },

    /**
     * Adds to the favourites storage the given contentId.
     *
     * @param contentId
     *   The content id to be stored as favourite.
     */
    addToFavs: function (contentId) {
      return Drupal.behaviors.simpleFavs.getFavs().then((favs) => {
        if (!Array.isArray(favs)) {
          favs = [];
        }

        if (!favs.includes(contentId)) {
          favs.push(contentId);
        }

        return Drupal.behaviors.simpleFavs.setFavs(favs);
      });
    },

    /**
     * Removes from the favourites storage the given contentId.
     *
     * @param contentId
     *   The content id to be removed as favourite.
     */
    removeFromFavs: function (contentId) {
      return Drupal.behaviors.simpleFavs.getFavs().then((favs) => {
        if (!Array.isArray(favs)) {
          favs = [];
        }

        const filtered = favs.filter(id => parseInt(id) !== parseInt(contentId));
        return Drupal.behaviors.simpleFavs.setFavs(filtered);
      });
    },

    /**
     * Removes the current non-node page from the favourites.
     */
    removeFromFavsOther: function () {
      return Drupal.behaviors.simpleFavs.getFavsOther().then((favs) => {
        if (!Array.isArray(favs)) {
          favs = [];
        }

        const path = window.location.pathname;
        const filtered = favs.filter(fav => fav.path !== path);
        return Drupal.behaviors.simpleFavs.setFavsOther(filtered);
      });
    },

    /**
     * Sets the favs cookie.
     */
    setCookie: function (cookieValue) {
      const cookieExpires = Drupal.behaviors.simpleFavs.getExpirationDate();
      document.cookie = 'content_favs=' + JSON.stringify(cookieValue) + ';expires=' + cookieExpires.toGMTString() + ';path=/';
    },

    /**
     * Defines the cookie expiration time based on the configuration.
     *
     * @returns {Date}
     *   The expiration date for the cookie.
     */
    getExpirationDate: function () {
      const now = new Date();
      const cookieLifetime = parseInt(
        drupalSettings.simple_favs?.views_heart?.cookie_expiration ??
        drupalSettings.simple_favs?.heart?.cookie_expiration
      );
      now.setDate(now.getDate() + cookieLifetime);
      return now;
    },

    /**
     * Retrieves the list of stored favourites.
     */
    getFavs: function () {
      const useDatabase =
        drupalSettings.simple_favs?.views_heart?.use_database_storage ??
        drupalSettings.simple_favs?.heart?.use_database_storage ?? false;
      const uid = parseInt(
        drupalSettings.simple_favs?.views_heart?.current_uid ??
        drupalSettings.simple_favs?.heart?.current_uid
      );

      if (useDatabase && uid !== 0) {
        return new Promise((resolve) => {
          $.ajax({
            url: '/simple-favs/get-user-favs',
            method: 'GET',
            success: function (response) {
              if (response && typeof response === 'object') {
                // Convert object of { "0": 10, "1": 11 } to array: [10, 11]
                const values = Object.values(response).map(id => parseInt(id));
                resolve(values);
              }
              else {
                resolve([]);
              }
            },
            error: function () {
              resolve([]);
            }
          });
        });
      }

      // Fallback to cookie
      return Promise.resolve((() => {
        const name = "content_favs=";
        const decodedCookie = decodeURIComponent(document.cookie);
        const ca = decodedCookie.split(';');

        for (let c of ca) {
          c = c.trim();
          if (c.startsWith(name)) {
            try {
              const parsed = JSON.parse(c.substring(name.length));
              // Ensure it's always an array.
              return Array.isArray(parsed) ? parsed : [];
            }
            catch (e) {
              return [];
            }
          }
        }
        return [];
      })());
    },

    /**
     * Writes the favourites list to DB or cookie depending on config.
     */
    setFavs: function (favs) {
      favs_orig = favs;
      favs = Object.assign({}, favs);
      const useDatabase =
        drupalSettings.simple_favs?.views_heart?.use_database_storage ??
        drupalSettings.simple_favs?.heart?.use_database_storage ?? false;
      const uid = parseInt(
        drupalSettings.simple_favs?.views_heart?.current_uid ??
        drupalSettings.simple_favs?.heart?.current_uid
      );

      if (useDatabase && uid !== 0) {
        return $.ajax({
          url: '/simple-favs/set-user-favs',
          method: 'POST',
          contentType: 'application/json',
          data: JSON.stringify(favs)
        });
      }

      const cookieExpires = Drupal.behaviors.simpleFavs.getExpirationDate();
      document.cookie = 'content_favs=' + JSON.stringify(favs_orig) +
        ';expires=' + cookieExpires.toGMTString() + ';path=/';

      return Promise.resolve();
    },

    /**
     * Retrieves non-node favourites (e.g., views, taxonomy).
     * Returns an array of { path, title }.
     */
    getFavsOther: function () {
      return new Promise((resolve) => {
        const useDatabase =
          drupalSettings.simple_favs?.views_heart?.use_database_storage ??
          drupalSettings.simple_favs?.heart?.use_database_storage ?? false;
        const uid = parseInt(
          drupalSettings.simple_favs?.views_heart?.current_uid ??
          drupalSettings.simple_favs?.heart?.current_uid
        );

        if (useDatabase && uid !== 0) {
          $.ajax({
            url: '/simple-favs/get-user-favs-other',
            method: 'GET',
            success: function (response) {
              // Defensive: ensure response is an array of objects with `path`
              if (Array.isArray(response)) {
                resolve(response);
              }
              else {
                console.warn('Unexpected response for getFavsOther:', response);
                resolve([]);
              }
            },
            error: function () {
              console.warn('Failed to load other route favourites.');
              resolve([]);
            }
          });
        }
        else {
          // Anonymous fallback — store in cookie.
          const name = 'content_favs_other=';
          const decodedCookie = decodeURIComponent(document.cookie);
          const ca = decodedCookie.split(';');
          for (let i = 0; i < ca.length; i++) {
            let c = ca[i].trim();
            if (c.indexOf(name) === 0) {
              try {
                const parsed = JSON.parse(c.substring(name.length));
                if (Array.isArray(parsed)) {
                  resolve(parsed);
                }
                else {
                  resolve([]);
                }
                return;
              }
              catch (e) {
                resolve([]);
                return;
              }
            }
          }

          resolve([]);
        }
      });
    },



    /**
     * Stores non-node favourites to DB or cookie.
     *
     * @param favsArray
     *   Array of { path, title } objects.
     */
    setFavsOther: function (favsArray) {
      const useDatabase =
        drupalSettings.simple_favs?.views_heart?.use_database_storage ??
        drupalSettings.simple_favs?.heart?.use_database_storage ?? false;
      const uid = parseInt(
        drupalSettings.simple_favs?.views_heart?.current_uid ??
        drupalSettings.simple_favs?.heart?.current_uid
      );

      if (useDatabase && uid !== 0) {
        $.ajax({
          url: '/simple-favs/set-user-favs-other',
          method: 'POST',
          contentType: 'application/json',
          data: JSON.stringify(favsArray),
          success: function () {
            console.log('Saved other route favourites to DB.');
          },
          error: function () {
            console.warn('Failed to save other route favourites to DB.');
          }
        });
      }
      else {
        const cookieExpires = Drupal.behaviors.simpleFavs.getExpirationDate();
        document.cookie = 'content_favs_other=' + JSON.stringify(favsArray) +
          ';expires=' + cookieExpires.toGMTString() + ';path=/';
        console.log('Saved other route favourites to cookie.');
      }
    },

    setFavsOtherFromViews: function (path) {
      // Send just the path, let the server resolve the title.
      const useDatabase =
        drupalSettings.simple_favs?.views_heart?.use_database_storage ??
        drupalSettings.simple_favs?.heart?.use_database_storage ?? false;
      const uid = parseInt(
        drupalSettings.simple_favs?.views_heart?.current_uid ??
        drupalSettings.simple_favs?.heart?.current_uid
      );

      if (useDatabase && uid !== 0) {
        return $.ajax({
          url: '/simple-favs/set-user-favs-other-from-views',
          method: 'POST',
          contentType: 'application/json',
          data: JSON.stringify({ path }),
          success: function () {
            console.log('Saved other route favourite using path (server-resolved title).');
          },
          error: function () {
            console.warn('Failed to save other route favourite via path.');
          }
        });
      }

      console.warn('setFavsOtherFromViews: not supported in cookie mode yet.');
      return Promise.resolve();
    },

    /**
     * Checks if the given contentId is in favourites or not.
     *
     * @param contentId
     *   The content id to be checked.
     *
     * @return boolean
     */
    isInFavs: function (contentId) {
      return Drupal.behaviors.simpleFavs.getFavs().then(favs => {
        return favs.includes(parseInt(contentId));
      });
    },

    /**
     * Checks if the current non-node page is in the favourites.
     *
     * @return {boolean}
     */
    isInFavsOther: function () {
      const currentPath = window.location.pathname.replace(/\/+$/, ''); // Strip trailing slash.
      return Drupal.behaviors.simpleFavs.getFavsOther().then(favs => {
        return favs.some(fav => fav.path.replace(/\/+$/, '') === currentPath);
      });
    },

    /**
     * Checks if the current non-node page is in the favourites.
     *
     * @param path
     *   The path to be checked.
     *
     * @return {boolean}
     */
    isInFavsOtherViews: function (path) {
      const currentPath = path;
      return Drupal.behaviors.simpleFavs.getFavsOther().then(favs => {
        return favs.some(fav => fav.path.replace(/\/+$/, '') === currentPath);
      });
    },
  };
})(jQuery, Drupal);

