<?php

namespace Drupal\simple_favs\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for managing favourites.
 */
class SimpleFavsManageController extends ControllerBase {

  /**
   * Returns the management page for favourites.
   */
  public function managePage() {
    $config = $this->config('simple_favs.settings');
    $max_display = (int) $config->get('block_max_display') ?? 8;
    $uid = $this->currentUser()->id();
    $use_db = (($uid === 0) ? FALSE : $config->get('use_database_storage')) ?? FALSE;

    $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();
    $favourites = [];

    if ($uid) {
      $connection = \Drupal::database();

      // Load node favourites.
      $node_result = $connection->select('simple_favs_user', 's')
        ->fields('s', ['favs'])
        ->condition('uid', $uid)
        ->execute()
        ->fetchField();

      if ($node_result) {
        $nids = unserialize($node_result);
        if (is_array($nids)) {
          $limited_nids = array_slice(array_filter($nids, 'is_numeric'), 0, 50);
          $nodes = \Drupal\node\Entity\Node::loadMultiple($limited_nids);

          foreach ($nodes as $nid => $node) {
            if ($node->access('view')) {
              if ($node->hasTranslation($langcode)) {
                $node = $node->getTranslation($langcode);
              }
              $favourites[] = [
                'path' => '/' . $langcode . '/node/' . $nid,
                'title' => $node->label(),
              ];
            }
          }
        }
        else {
          \Drupal::logger('simple_favs')->warning('Could not unserialize node favourites for user @uid.', ['@uid' => $uid]);
        }
      }

      // Load non-node favourites.
      $query = $connection->select('simple_favs_user_other', 'o')
        ->fields('o', ['path', 'title'])
        ->condition('uid', $uid)
        ->range(0, 50);

      foreach ($query->execute() as $record) {
        $favourites[] = [
          'path' => $record->path,
          'title' => $record->title,
        ];
      }
    }

    $build = [
      '#theme' => 'simple_favs_manage_page',
      '#attached' => [
        'library' => [
          'simple_favs/manage_page',
        ],
        'drupalSettings' => [
          'simple_favs' => [
            'manage' => [
              'max_display' => $max_display,
              'favs' => $favourites,
	      'current_uid' => $uid,
              'langcode' => $langcode,
              'use_database_storage' => $use_db,
            ],
          ],
        ],
      ],
      '#cache' => [
        'tags' => [
          'config:simple_favs.settings',
          'simple_favs:block',
          'simple_favs:user:' . $uid,
        ],
        'contexts' => [
          'route',
          'user',
          'languages:language_interface',
        ],
      ],
    ];

    return $build;
  }

  /**
   * Saves updated titles for a user.
   */
  public function saveTitles(Request $request) {
    $data = json_decode($request->getContent(), TRUE);
    $uid = (int) $this->currentUser()->id();
    $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();

    if (!$uid || !isset($data['favourites']) || !is_array($data['favourites'])) {
      return new JsonResponse(['status' => 'error', 'message' => 'Invalid data'], 400);
    }

    $connection = \Drupal::database();

    // Load existing node favs.
    $node_result = $connection->select('simple_favs_user', 's')
      ->fields('s', ['favs'])
      ->condition('uid', $uid)
      ->execute()
      ->fetchField();

    $existing_node_favs = [];
    if ($node_result) {
      $existing_node_favs = unserialize($node_result);
      if (!is_array($existing_node_favs)) {
        $existing_node_favs = [];
      }
    }

    $updated_node_favs = [];
    $updated_other_favs = [];

    foreach ($data['favourites'] as $fav) {
      $path = $fav['path'] ?? '';
      $title = $fav['title'] ?? '';

      if (preg_match('#^/[^/]+/node/(\d+)$#', $path, $matches)) {
        $updated_node_favs[] = (int) $matches[1];
      }
      else {
        if (!empty($path)) {
          $updated_other_favs[] = [
            'path' => $path,
            'title' => $title,
            'langcode' => $langcode,
          ];
        }
      }
    }

    // Update node favourites.
    $connection->merge('simple_favs_user')
      ->key(['uid' => $uid])
      ->fields(['favs' => serialize($updated_node_favs)])
      ->execute();

    // Replace non-node favourites.
    $connection->delete('simple_favs_user_other')
      ->condition('uid', $uid)
      ->execute();

    foreach ($updated_other_favs as $fav) {
      $connection->insert('simple_favs_user_other')
        ->fields([
          'uid' => $uid,
          'path' => $fav['path'],
          'title' => $fav['title'],
          'langcode' => $fav['langcode'],
        ])
        ->execute();
    }

    \Drupal::service('cache_tags.invalidator')->invalidateTags(['simple_favs:user:' . $uid]);
    return new JsonResponse(['status' => 'ok']);
  }

}

