<?php

namespace Drupal\simple_favs\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Session\AccountProxyInterface;

class SimpleFavsStorageController extends ControllerBase {

  protected $currentUser;
  protected $database;

  public function __construct(AccountProxyInterface $current_user, Connection $database) {
    $this->currentUser = $current_user;
    $this->database = $database;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('current_user'),
      $container->get('database')
    );
  }

  /**
   * Returns the favourites array for the current user.
   */
  public function getUserFavs() {
    if ($this->currentUser->isAnonymous()) {
      throw new AccessDeniedHttpException('Anonymous users must use cookie-based storage.');
    }

    $uid = $this->currentUser->id();
    $result = $this->database->select('simple_favs_user', 'f')
      ->fields('f', ['favs'])
      ->condition('uid', $uid)
      ->execute()
      ->fetchField();

    $favs = $result ? unserialize($result) : [];
    return new JsonResponse($favs);
  }

  /**
   * Stores the favourites array for the current user.
   */
  public function setUserFavs(Request $request) {
    if ($this->currentUser->isAnonymous()) {
      throw new AccessDeniedHttpException('Anonymous users must use cookie-based storage.');
    }

    $favs = json_decode($request->getContent(), TRUE);
    if (!is_array($favs)) {
      return new JsonResponse(['error' => 'Invalid format'], 400);
    }

    $uid = $this->currentUser->id();
    $this->database->merge('simple_favs_user')
      ->key(['uid' => $uid])
      ->fields(['favs' => serialize($favs)])
      ->execute();

    \Drupal::service('cache_tags.invalidator')->invalidateTags(['simple_favs:user:' . $uid]);
    return new JsonResponse(['success' => TRUE]);
  }

}

