<?php

namespace Drupal\simple_favs\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Session\AccountProxyInterface;

class SimpleFavsStorageOtherController extends ControllerBase {

  protected $currentUser;
  protected $database;

  public function __construct(AccountProxyInterface $current_user, Connection $database) {
    $this->currentUser = $current_user;
    $this->database = $database;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('current_user'),
      $container->get('database')
    );
  }

  public function getUserFavsOther(Request $request) {
    $uid = \Drupal::currentUser()->id();
    $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();
    $favs = [];

    if ($uid) {
      $query = \Drupal::database()->select('simple_favs_user_other', 'f')
        ->fields('f', ['path', 'title'])
        ->condition('uid', $uid)
        ->condition('langcode', $langcode);
      $results = $query->execute();
      foreach ($results as $record) {
        $favs[] = ['path' => $record->path, 'title' => $record->title];
      }
    }

    return new JsonResponse($favs);
  }

  public function setUserFavsOther(Request $request) {
    $uid = \Drupal::currentUser()->id();
    $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();

    if ($uid) {
      $data = json_decode($request->getContent(), TRUE);

      if (is_array($data)) {
        \Drupal::database()->delete('simple_favs_user_other')
          ->condition('uid', $uid)
          ->condition('langcode', $langcode)
          ->execute();

        foreach ($data as $entry) {
          if (!empty($entry['path']) && isset($entry['title'])) {
            \Drupal::database()->insert('simple_favs_user_other')
              ->fields([
                'uid' => $uid,
                'path' => $entry['path'],
                'title' => $entry['title'],
                'langcode' => $langcode,
              ])->execute();
          }
        }
      }
    }

    \Drupal::service('cache_tags.invalidator')->invalidateTags(['simple_favs:user:' . $uid]);
    return new JsonResponse(['status' => 'ok']);
  }

  public function setUserFavsOtherFromViews(Request $request) {
    $uid = $this->currentUser->id();
    $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();

    if ($uid === 0) {
      throw new AccessDeniedHttpException();
    }

    $data = json_decode($request->getContent(), TRUE);
    $path = $data['path'] ?? NULL;

    if (empty($path)) {
      return new JsonResponse(['error' => 'Missing path.'], 400);
    }

    // Check for existing entry.
    $exists = $this->database->select('simple_favs_user_other', 'f')
      ->fields('f')
      ->condition('uid', $uid)
      ->condition('langcode', $langcode)
      ->condition('path', $path)
      ->execute()
      ->fetchField();

    if (!$exists) {
      $title = $this->resolveTitleFromPath($path);
      if (!$title) {
        return new JsonResponse(['error' => 'Unable to resolve title.'], 400);
      }

      $this->database->insert('simple_favs_user_other')
        ->fields([
          'uid' => $uid,
          'path' => $path,
          'title' => $title,
          'langcode' => $langcode,
        ])->execute();

      \Drupal::service('cache_tags.invalidator')->invalidateTags(['simple_favs:user:' . $uid]);
    }

    return new JsonResponse(['status' => 'ok']);
  }

  protected function resolveTitleFromPath(string $path): ?string {
    $alias_manager = \Drupal::service('path_alias.manager');
    $internal_path = $alias_manager->getPathByAlias($path);

    // Node title resolution.
    if (preg_match('/^\/?node\/(\d+)$/', $internal_path, $matches)) {
      $node = \Drupal\node\Entity\Node::load($matches[1]);
      return $node?->label();
    }

    // Taxonomy title resolution.
    if (preg_match('/^\/?taxonomy\/term\/(\d+)$/', $internal_path, $matches)) {
      $term = \Drupal\taxonomy\Entity\Term::load($matches[1]);
      return $term?->label();
    }

    // Otherwise fallback to last path segment or raw path.
    return urldecode(basename($path));
  }

}

