<?php

namespace Drupal\simple_favs\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Language\LanguageInterface;

/**
 * Class SimpleFavsSettingsForm.
 */
class SimpleFavsSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'simple_favs.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'simple_favs_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('simple_favs.settings');
    $languages = \Drupal::languageManager()->getLanguages(LanguageInterface::STATE_CONFIGURABLE);
    $default_langcode = \Drupal::languageManager()->getDefaultLanguage()->getId();

    $form['block_display'] = [
      '#type' => 'details',
      '#title' => $this->t('Block display settings'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];
    // Translatable values wrapped in tabs.
    $form['block_display']['translatables'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Language-specific block settings'),
    ];

    foreach ($languages as $langcode => $language) {
      $form['block_display'][$langcode] = [
        '#type' => 'details',
        '#title' => $language->getName(),
        '#group' => 'translatables',
      ];

      $lang_config = \Drupal::languageManager()->getLanguageConfigOverride($langcode, 'simple_favs.settings');

      $form['block_display'][$langcode]['heart_tooltip_text'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Heart icon tooltip text for add to my favourites'),
        '#default_value' => $lang_config->get('heart_tooltip_text') ?? $config->get('heart_tooltip_text') ?? ($langcode === 'fr' ? 'Ajouter à mes favoris' : 'Add to my favourites'),
        '#description' => $this->t('This text will be used as a tooltip and aria-label on the heart icon when the item has not yet been chosen as a favourite.'),
      ];
      $form['block_display'][$langcode]['heart_tooltip_remove_text'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Heart icon tooltip text for remove from my favourites'),
        '#default_value' => $lang_config->get('heart_tooltip_remove_text') ?? $config->get('heart_tooltip_remove_text') ?? ($langcode === 'fr' ? 'Retirer de mes favoris' : 'Remove from my favourites'),
        '#description' => $this->t('This text will be used as a tooltip and aria-label on the heart icon when the item is currently in the list of favourites.'),
      ];
      $form['block_display'][$langcode]['wrapper_prefix'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Wrapper prefix'),
        '#description' => $this->t('Optional markup to add before the title and fallback.'),
        '#default_value' => $lang_config->get('wrapper_prefix') ?? $config->get('wrapper_prefix') ?? '',
      ];

      $form['block_display'][$langcode]['title_prefix'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Title prefix'),
        '#default_value' => $lang_config->get('title_prefix') ?? $config->get('title_prefix') ?? '',
      ];

      $form['block_display'][$langcode]['title_suffix'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Title suffix'),
        '#default_value' => $lang_config->get('title_suffix') ?? $config->get('title_suffix') ?? '',
      ];

      $form['block_display'][$langcode]['fallback_prefix'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Fallback prefix'),
        '#default_value' => $lang_config->get('fallback_prefix') ?? $config->get('fallback_prefix') ?? '',
      ];
      $form['block_display'][$langcode]['fallback_suffix'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Fallback suffix'),
        '#default_value' => $lang_config->get('fallback_suffix') ?? $config->get('fallback_suffix') ?? '',
      ];

      $form['block_display'][$langcode]['wrapper_suffix'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Wrapper suffix'),
        '#description' => $this->t('Optional markup to add after the title and fallback.'),
        '#default_value' => $lang_config->get('wrapper_suffix') ?? $config->get('wrapper_suffix') ?? '',
      ];

      $form['block_display'][$langcode]['title_classes'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Additional title classes'),
        '#description' => $this->t('Space-separated CSS class names to add to the &lt;h2&gt; title element.'),
        '#default_value' => $lang_config->get('title_classes') ?? $config->get('title_classes') ?? '',
      ];

      $form['block_display'][$langcode]['favs_list_classes'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Favourites list classes'),
        '#default_value' => $lang_config->get('favs_list_classes') ?? $config->get('favs_list_classes'),
        '#description' => $this->t('Additional classes to apply to the &lt;ul&gt; element in the My favourites block. Separate with spaces.'),
      ];
    }

    $form['cookie_expiration'] = [
      '#type' => 'number',
      '#title' => $this->t('Cookie expiration'),
      '#description' => $this->t('Please, set how many days the cookie with the favourites information should be stored.'),
      '#default_value' => $config->get('cookie_expiration') ?? 7200,
    ];

    $form['block_max_items'] = [
      '#type' => 'number',
      '#title' => $this->t('Max number of favourites to display in the My favourites block.'),
      '#description' => $this->t('The default value is 8.'),
      '#default_value' => $config->get('block_max_items') ?? 8,
    ];

    $form['use_database_storage'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use database storage instead of cookies'),
      '#default_value' => $config->get('use_database_storage') ?? TRUE,
      '#description' => $this->t('If enabled, favourites will be stored in the database for authenticated users. Anonymous users will still use cookies.'),
    ];

    $form['fontawesome_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Font Awesome heart'),
      '#default_value' => $config->get('fontawesome_enabled') ?? \Drupal::moduleHandler()->moduleExists('fontawesome'),
      '#description' => $this->t('Use a Font Awesome heart icon instead of text-only fallback.'),
      '#disabled' => !\Drupal::moduleHandler()->moduleExists('fontawesome'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    // Save global (non-translatable) settings.
    $this->config('simple_favs.settings')
      ->set('cookie_expiration', $form_state->getValue('cookie_expiration'))
      ->set('block_max_items', $form_state->getValue('block_max_items'))
      ->set('use_database_storage', $form_state->getValue('use_database_storage'))
      ->set('fontawesome_enabled', $form_state->getValue('fontawesome_enabled'))
      ->save();

    // Save translated values.
    $languages = \Drupal::languageManager()->getLanguages(LanguageInterface::STATE_CONFIGURABLE);

    foreach ($languages as $langcode => $language) {
      $config_override = \Drupal::languageManager()->getLanguageConfigOverride($langcode, 'simple_favs.settings');

      $config_override
        ->set('heart_tooltip_text', $form_state->getValue(['block_display', $langcode, 'heart_tooltip_text']))
        ->set('heart_tooltip_remove_text', $form_state->getValue(['block_display', $langcode, 'heart_tooltip_remove_text']))
        ->set('wrapper_prefix', $form_state->getValue(['block_display', $langcode, 'wrapper_prefix']))
        ->set('title_prefix', $form_state->getValue(['block_display', $langcode, 'title_prefix']))
        ->set('title_suffix', $form_state->getValue(['block_display', $langcode, 'title_suffix']))
        ->set('fallback_prefix', $form_state->getValue(['block_display', $langcode, 'fallback_prefix']))
        ->set('fallback_suffix', $form_state->getValue(['block_display', $langcode, 'fallback_suffix']))
        ->set('wrapper_suffix', $form_state->getValue(['block_display', $langcode, 'wrapper_suffix']))
        ->set('title_classes', $form_state->getValue(['block_display', $langcode, 'title_classes']))
        ->set('favs_list_classes', $form_state->getValue(['block_display', $langcode, 'favs_list_classes']))
        ->save();
    }
  }

}

