<?php

namespace Drupal\simple_favs\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\node\Entity\Node;

/**
 * Provides a "My favourites" block for Simple Favs.
 *
 * @Block(
 *   id = "simple_favs_block",
 *   admin_label = @Translation("Simple Favs"),
 * )
 */
class SimpleFavsBlock extends BlockBase {

  /**
   * {@inheritdoc}
   */
  public function build() {
    // Get the interface language ID.
    $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();

    $config = \Drupal::config('simple_favs.settings');
    // Load language-specific config override.
    $config_lang = \Drupal::languageManager()->getLanguageConfigOverride($langcode, 'simple_favs.settings');
    // Add "Manage favourites" link if there are items.
    $manage_url = Url::fromRoute('simple_favs.manage', [], ['absolute' => TRUE])->toString();
    $manage_link_markup = '<div class="text-center mrgn-tp-sm button-wrapper center-text"><a class="btn btn-default" href="' . $manage_url . '">' . $this->t('Manage favourites') . '</a></div>';

    $wrapper_prefix = $config_lang->get('wrapper_prefix') ?? '';
    $wrapper_suffix = $config_lang->get('wrapper_suffix') ?? '';
    $title_prefix = $config_lang->get('title_prefix') ?? '';
    $title_suffix = $config_lang->get('title_suffix') ?? '';
    $fallback_prefix = $config_lang->get('fallback_prefix') ?? '';
    $fallback_suffix = $config_lang->get('fallback_suffix') ?? '';
    $title_classes = $config_lang->get('title_classes') ?? '';
    $favs_list_classes = $config_lang->get('favs_list_classes') ?? '';

    // Safe fallback for language (supports single-language sites).
    $langcode = 'und';
    if (\Drupal::hasService('language_manager')) {
      $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();
    }

    $uid = (int) \Drupal::currentUser()->id();
    $use_db = (($uid === 0) ? FALSE : $config->get('use_database_storage')) ?? FALSE;
    $block_max_items = $config->get('block_max_items') ?? 8;

    $favs = [];
    $favs_other = [];
    $list_markup = '';

    if ($uid && $use_db) {
      $connection = \Drupal::database();
      $result = $connection->select('simple_favs_user', 's')
        ->fields('s', ['favs'])
        ->condition('uid', $uid)
        ->execute()
        ->fetchField();

      if ($result) {
        $favs = unserialize($result);
      }

      $query = $connection->select('simple_favs_user_other', 'o')
        ->fields('o', ['path', 'title'])
        ->condition('uid', $uid)
        ->execute();

      foreach ($query as $record) {
        $favs_other[] = ['path' => $record->path, 'title' => $record->title];
      }
    }

    if (!empty($favs) || !empty($favs_other)) {
      $items = [];

      foreach ($favs as $nid) {
        $node = Node::load($nid);
        if ($node && $node->access('view')) {
          $url = '/node/' . $nid;
          if ($node->hasTranslation($langcode)) {
            $node = $node->getTranslation($langcode);
            $url = '/' . $langcode . '/node/' . $nid;
          }
          $title = $node->label();
          $items[] = '<li><a href="' . $url . '">' . $title . '</a></li>';
        }
      }

      foreach ($favs_other as $fav) {
        $path = $fav['path'] ?? '';
        $title = $fav['title'] ?? '';
        if (!empty($path) && !empty($title)) {
          $items[] = '<li><a href="' . $path . '">' . $title . '</a></li>';
        }
      }
      $items_maxed = [];
      $count = 0;

      foreach ($items as $item) {
        if ($count > $block_max_items) {
          break;
        }
        $items_maxed[] = $item;
        $count++;
      }

      if (!empty($items)) {
        $list_markup = '<ul class="simple-favs-list ' . $favs_list_classes . '">' . implode('', $items_maxed) . '</ul>';
      }
    }

    if (empty($list_markup)) {
      $list_markup = '<p class="' . $favs_list_classes . '">' . $this->t("You don't have any favourites at the moment.") . '</p>';
    }
    elseif (!$uid) {
      // Anonymous favourite management needs page needs work, disable link for now.
      $manage_link_markup = '';
    }
    elseif ($uid > 0) {
      $list_markup .= $manage_link_markup;
    }

    return [
      '#type' => 'container',
      '#attributes' => [
        'data-block-plugin-id' => 'simple-favs-block',
        'class' => ['simple-favs-wrapper'],
      ],
      'prefix' => ['#markup' => $wrapper_prefix],
      'title' => [
        '#markup' => $title_prefix . '<h2 class="simple-favs ' . $title_classes . '">' . $this->t('My favourites') . '</h2>' . $title_suffix,
      ],
      'favs' => [
        '#markup' => $fallback_prefix . $list_markup . $fallback_suffix,
      ],
      'suffix' => ['#markup' => $wrapper_suffix],
      '#cache' => [
        'contexts' => $this->getCacheContexts(),
        'tags' => $this->getCacheTags(),
      ],
      '#attached' => [
        'library' => ['simple_favs/favs'],
        'drupalSettings' => [
          'simple_favs' => [
            'block' => [
              'title_prefix' => $title_prefix,
              'title_suffix' => $title_suffix,
              'fallback_prefix' => $fallback_prefix,
              'fallback_suffix' => $fallback_suffix,
              'wrapper_prefix' => $wrapper_prefix,
              'wrapper_suffix' => $wrapper_suffix,
              'title_classes' => $title_classes,
              'favs_list_classes' => $favs_list_classes,
              'manage_link_markup' => $manage_link_markup,
              'max_items' => $block_max_items,
	      'current_uid' => $uid,
	      'use_db' => $use_db,
              'langcode' => $langcode,
            ],
          ],
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags() {
    $tags = ['config:simple_favs.settings', 'simple_favs:block'];
    $uid = \Drupal::currentUser()->id();

    if ($uid) {
      $tags[] = 'simple_favs:user:' . $uid;
    }

    return $tags;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts() {
    return ['route', 'user', 'languages:language_interface'];
  }

}
