<?php

namespace Drupal\simple_favs\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Language\LanguageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Url;
use Drupal\path_alias\AliasManagerInterface;
use Drupal\node\Entity\Node;

/**
 * Provides a block with a heart icon to toggle favourites.
 *
 * @Block(
 *   id = "simple_favs_heart_block",
 *   admin_label = @Translation("Simple Favs Heart Icon"),
 *   context_definitions = {
 *     "node" = @ContextDefinition("entity:node", required = FALSE)
 *   }
 * )
 */
class SimpleFavsHeartBlock extends BlockBase implements ContainerFactoryPluginInterface {
  use StringTranslationTrait;

  protected RouteMatchInterface $routeMatch;
  protected AliasManagerInterface $aliasManager;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, RouteMatchInterface $route_match, AliasManagerInterface $alias_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->routeMatch = $route_match;
    $this->aliasManager = $alias_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_route_match'),
      $container->get('path_alias.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    // Get the interface language ID.
    $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();

    // Load language-specific config override.
    $config_lang = \Drupal::languageManager()->getLanguageConfigOverride($langcode, 'simple_favs.settings');

    $config = \Drupal::config('simple_favs.settings');
    $expiration = $config->get('cookie_expiration');
    $fontawesome = $config->get('fontawesome_enabled') && \Drupal::moduleHandler()->moduleExists('fontawesome');
    $uid = \Drupal::currentUser()->id();
    $use_db = ($uid === 0) ? FALSE : $config->get('use_database_storage');

    $langcode = 'und';
    if (\Drupal::hasService('language_manager')) {
      $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();
    }

    $node = $this->routeMatch->getParameter('node');
    $id = ($node && $node->id()) ? $node->id() : 0;
    $active = FALSE;

    if ($uid && $use_db) {
      $connection = \Drupal::database();

      if ($id) {
        $result = $connection->select('simple_favs_user', 's')
          ->fields('s', ['favs'])
          ->condition('uid', $uid)
          ->execute()
          ->fetchField();

        if ($result) {
          $favs = unserialize($result);
          $active = in_array($id, $favs);
        }
      }
      else {
        $current_path = \Drupal::service('path.current')->getPath();
        $current_alias = $this->aliasManager->getAliasByPath($current_path);

        $query = $connection->select('simple_favs_user_other', 'o')
          ->fields('o', ['path'])
          ->condition('uid', $uid)
          ->condition('langcode', $langcode);

        $results = $query->execute();
        foreach ($results as $record) {
          if ($record->path === $current_path || $record->path === $current_alias) {
            $active = TRUE;
            break;
          }
        }
      }
    }

    $tooltip_add = $config_lang->get('heart_tooltip_text') ?? $this->t('Add to my favourites');
    $tooltip_remove = $config_lang->get('heart_tooltip_remove_text') ?? $this->t('Remove from my favourites');
    $title = $active
      ? t('Remove from my favourites')
      : $tooltip_add;

    $icon_type = $fontawesome ? ' font-awesome' : ' character-mode';
    $icon_markup = $fontawesome
      ? ($active ? '<i class="fas fa-heart"></i>' : '<i class="far fa-heart"></i>')
      : ($active ? '♥' : '♡');

    return [
      '#markup' => '<div class="simple-favs-heart ' . ($active ? ' active ' : '') . $icon_type . '" data-id="' . $id . '" role="button" tabindex="0" aria-label="' . $title . '" title="' . $title . '"><span class="heart-icon">' . $icon_markup . '</span></div>',
      '#attached' => [
        'library' => ['simple_favs/heart'],
        'drupalSettings' => [
          'simple_favs' => [
            'heart' => [
              'cookie_expiration' => $expiration,
              'current_uid' => $uid,
              'fontawesome_enabled' => $fontawesome,
              'tooltip_add' => $tooltip_add,
              'tooltip_remove' => $tooltip_remove,
              'use_database_storage' => $use_db,
            ],
          ],
        ],
      ],
      '#cache' => [
        'contexts' => $this->getCacheContexts(),
        'tags' => $this->getCacheTags(),
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags() {
    $tags = ['config:simple_favs.settings', 'simple_favs:block'];
    $uid = \Drupal::currentUser()->id();

    if ($uid) {
      $tags[] = 'simple_favs:user:' . $uid;
    }

    return $tags;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts() {
    return ['route', 'user', 'languages:language_interface'];
  }

}
