<?php

namespace Drupal\simple_favs\Plugin\views\field;

use Drupal\Core\Render\Markup;
use Drupal\Core\Url;
use Drupal\views\Attribute\ViewsField;
use Drupal\views\ResultRow;
use Drupal\views\Plugin\views\field\FieldPluginBase;

/**
 * A handler to provide custom displays for subscription dates.
 *
 * @ingroup views_field_handlers
 */
#[ViewsField("simple_favs_views_heart")]
class SimpleFavsViewsHeart extends FieldPluginBase {

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $row) {
    $nid = NULL;

    // Case 1: Standard Views using node base table.
    if (isset($row->_entity) && is_object($row->_entity) && method_exists($row->_entity, 'id')) {
      $nid = $row->_entity->id();
    }

    // Case 2: Search API result row, only if the module is enabled.
    elseif (\Drupal::moduleHandler()->moduleExists('search_api') && isset($row->_item)) {
      try {
        $original_object = $row->_item->getOriginalObject();
        if ($original_object && method_exists($original_object, 'getValue')) {
          $entity = $original_object->getValue();
          if ($entity && method_exists($entity, 'id')) {
            $nid = $entity->id();
          }
        }
      }
      catch (\Throwable $e) {
        \Drupal::logger('simple_favs')->warning('Error extracting nid from Search API row: @error', ['@error' => $e->getMessage()]);
      }
    }

    if (!$nid) {
      \Drupal::logger('simple_favs')->warning('Could not extract nid from row: @debug', [
        '@debug' => serialize($row),
      ]);
      return ['#markup' => 'No node found.'];
    }
    $uid = \Drupal::currentUser()->id();
    $config = \Drupal::config('simple_favs.settings');
    $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();
    // Load language-specific config override.
    $config_lang = \Drupal::languageManager()->getLanguageConfigOverride($langcode, 'simple_favs.settings');
    $tooltip_add = $config_lang->get('heart_tooltip_text') ?? $this->t('Add to my favourites');
    $tooltip_remove = $config_lang->get('heart_tooltip_remove_text') ?? $this->t('Remove from my favourites');
    $use_db = (($uid === 0) ? FALSE : $config->get('use_database_storage')) ?? FALSE;
    $fontawesome = $config->get('fontawesome_enabled') && \Drupal::moduleHandler()->moduleExists('fontawesome');

    $active = FALSE;

    if ($uid && $use_db && $nid) {
      $result = \Drupal::database()
        ->select('simple_favs_user', 's')
        ->fields('s', ['favs'])
        ->condition('uid', $uid)
        ->execute()
        ->fetchField();

      if ($result) {
        $favs = unserialize($result);
        $active = in_array($nid, $favs);
      }
    }
    $title = $active
      ? $tooltip_remove
      : $tooltip_add;

    $front_path = Url::fromRoute('<front>', [], ['absolute' => FALSE])->toString();
    $front_raw = Url::fromRoute('<front>', [], ['absolute' => TRUE])->toString();
    $front_scheme_and_domain = str_replace($front_path, '', $front_raw);

    $icon_type = $fontawesome ? ' font-awesome' : ' character-mode';
    $icon_markup = $fontawesome
      ? ($active ? '<i class="fas fa-heart"></i>' : '<i class="far fa-heart"></i>')
      : ($active ? '♥' : '♡');

    return [
      '#markup' => Markup::create(
        '<div class="simple-favs-views views-heart ' . ($active ? 'active ' : ' ') . $icon_type . '" data-id="' . $nid . '" role="button" tabindex="0" aria-label="' . $title . '" title="' . $title . '"><span class="heart-icon">' . $icon_markup . '</span></div>'
      ),
      '#attached' => [
        'library' => ['simple_favs/views_heart'],
        'drupalSettings' => [
          'simple_favs' => [
            'views_heart' => [
              'cookie_expiration' => $config->get('cookie_expiration'),
              'current_uid' => $uid,
              'fontawesome_enabled' => $fontawesome,
              'front_path' => $front_path,
              'front_raw' => $front_raw,
              'front_scheme_and_domain' => $front_scheme_and_domain,
              'tooltip_add' => $tooltip_add,
              'tooltip_remove' => $tooltip_remove,
              'use_database_storage' => $use_db,
            ],
          ],
        ],
      ],
      '#cache' => [
        'contexts' => $this->getCacheContexts(),
        'tags' => $this->getCacheTags(),
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags() {
    $tags = [
      'config:simple_favs.settings',
      'simple_favs:block',
    ];

    $uid = \Drupal::currentUser()->id();
    if ($uid) {
      $tags[] = 'simple_favs:user:' . $uid;
    }

    return $tags;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts() {
    return ['route', 'user', 'languages:language_interface'];
  }

}

