<?php

namespace Drupal\simple_like_button\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CssCommand;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Component\Utility\Html;
use Drupal\simple_like_button\Entity\SimpleLike;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityInterface;

/**
 * Class LikeForm.
 */
class LikeForm extends FormBase {

  /**
   * Returns the current_route_match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $route;

  /**
   * Returns the database service.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Form constructor.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route
   *   Provides an interface for classes representing the result of routing.
   * @param \Drupal\Core\Database\Connection $database
   *   Base Database API class.
   */
  public function __construct(RouteMatchInterface $route, Connection $database) {
    $this->route = $route;
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('current_route_match'),
      $container->get('database')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'like_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $current_entity = $this->getCurrentEntity();
    $logged_in = $this->currentUser()->isAuthenticated();
    // If user is not logged in, or current page is no entity, return null.
    if (!$logged_in || !$current_entity) {
      return NULL;
    }
    $entity = $current_entity->getEntityTypeId();
    $bundle = $current_entity->bundle();
    $entity_id = $current_entity->id();
    $uid = $this->currentUser()->id();
    $like_status = $this->getLikeStatus($entity, $entity_id, $bundle, $uid);
    $like_count = $this->getLikeCount($entity, $entity_id, $bundle);
    $liked_class = is_numeric($like_status) ? $this->t('liked') : '';
    $button_text = is_numeric($like_status) ? $this->t('Liked ·') . ' ' . $like_count : $this->t('Like ·') . ' ' . $like_count;
    $users_that_liked = ($like_count > 0) ? $this->getUsersThatLiked($entity, $entity_id, $bundle, $uid) : '';

    $form['entity'] = [
      '#type' => 'hidden',
      '#required' => TRUE,
      '#default_value' => $entity,
    ];

    $form['entity_id'] = [
      '#type' => 'hidden',
      '#required' => TRUE,
      '#default_value' => $entity_id,
    ];

    $form['bundle'] = [
      '#type' => 'hidden',
      '#required' => TRUE,
      '#default_value' => $bundle,
    ];

    $form['actions'] = [
      '#prefix' => '<span class="' . $entity . $entity_id . ' ' . $liked_class . '"> ',
      '#type' => 'button',
      '#attributes' => ['class' => [$entity . $entity_id]],
      '#value' => $button_text,
      '#ajax' => [
        'callback' => '::submitLikeAjax',
        'disable-refocus' => TRUE,
        'progress' => [
          'type' => 'none',
        ],
      ],
      '#suffix' => '</span>',
    ];

    if (!empty($users_that_liked)) {
      $form['users_that_liked'] = [
        '#type' => 'markup',
        '#weight' => '50',
        '#markup' => '<span class="users_that_liked">' . $this->t('Liked by:') . ' ' . $users_that_liked . '</span>',
      ];
    }
    $form['#attached']['library'][] = 'simple_like_button/like';

    return $form;
  }

  /**
   * Ajax callback to validate the email field.
   */
  public function submitLikeAjax(array &$form, FormStateInterface $form_state) {
    // Initiate.
    $response = new AjaxResponse();
    $uid = $this->currentUser()->id();
    $entity = $form_state->getValue('entity');
    $entity_id = $form_state->getValue('entity_id');
    $bundle = $form_state->getValue('bundle');
    $like_count = $this->getLikeCount($entity, $entity_id, $bundle);
    $like_count = empty($like_count) ? 0 : $like_count;
    $like_status = $this->getLikeStatus($entity, $entity_id, $bundle, $uid);

    // Record exists, unlike, remove record.
    if (is_numeric($like_status)) {
      $this->database->delete('simple_like')
        ->condition('entity', $entity, '=')
        ->condition('entity_id', $entity_id, '=')
        ->condition('bundle', $bundle, '=')
        ->condition('user_id', $uid, '=')
        ->execute();

      // Add ajax commands.
      $css_you_liked = ['display' => 'none'];
      $response->addCommand(new CssCommand('.you_liked_' . $entity . $entity_id, $css_you_liked));
      $css = ['color' => '#9f9c9c'];
      $response->addCommand(new CssCommand('.' . $entity . $entity_id, $css));
      $new_count = strval($like_count - 1);
      $response->addCommand(new InvokeCommand('.' . $entity . $entity_id, 'val', [t('Like ·') . ' ' . $new_count]));
      $response->addCommand(new HtmlCommand('#like_count_' . $entity . $entity_id, $new_count));

    }
    // Record doesn't exists, Like, add record.
    elseif (empty($like_status)) {
      $simple_like = SimpleLike::create([
        'entity' => Html::escape($entity),
        'entity_id' => $entity_id,
        'bundle' => $bundle,
      ]);

      $simple_like->save();
      // Add ajax commands.
      $response->addCommand(new HtmlCommand('.you_liked_' . $entity . $entity_id, 'You, '));
      $css = ['color' => '#0037fd'];
      $response->addCommand(new CssCommand('.' . $entity . $entity_id, $css));
      $css_you_liked = ['display' => 'initial'];
      $response->addCommand(new CssCommand('.you_liked_' . $entity . $entity_id, $css_you_liked));
      $new_count = strval($like_count + 1);
      $response->addCommand(new InvokeCommand('.' . $entity . $entity_id, 'val', [$this->t('Liked ·') . ' ' . $new_count]));
      $response->addCommand(new HtmlCommand('#like_count_' . $entity . $entity_id, $new_count));
    }

    // Wipe all messages, so on page refresh nothing comes up.
    $this->messenger()->deleteAll();
    return $response;
  }

  /**
   * Gets the user who made a like.
   */
  private function getUsersThatLiked($entity, $entity_id, $bundle, $uid) {
    // Query users that liked.
    $query = $this->database->select('simple_like', 'sil');
    $query->addField('sil', 'user_id');
    $query->addField('ufd', 'name');
    $query->condition('sil.entity', $entity);
    $query->condition('sil.bundle', $bundle);
    $query->condition('sil.entity_id', $entity_id);
    $query->join('users_field_data', 'ufd', 'ufd.uid = sil.user_id');
    $users = $query->execute()->fetchAll();
    $total = count((array) $users);

    // Build users string.
    $users_string = '';
    $liked_by_current = '';
    $i = 0;
    foreach ($users as $user) {
      if ($user->user_id == $uid) {
        $liked_by_current = $this->t('You,') . ' ';
        continue;
      }
      $users_string .= $user->name;
      if (++$i === $total) {
        $users_string .= '.';
      }
      else {
        $users_string .= ', ';
      }
    }
    return '<span class="you_liked you_liked_' . $entity . $entity_id . '">' . $liked_by_current . '</span>' . $users_string;
  }

  /**
   * Gets the like status.
   */
  private function getLikeStatus($entity, $entity_id, $bundle, $uid) {
    $query = $this->database->select('simple_like', 'sil');
    $query->addField('sil', 'id');
    $query->condition('sil.entity', $entity);
    $query->condition('sil.entity_id', $entity_id);
    $query->condition('sil.bundle', $bundle);
    $query->condition('sil.user_id', $uid);

    return $query->execute()->fetchField();
  }

  /**
   * Gets the like count.
   */
  private function getLikeCount($entity, $entity_id, $bundle) {
    $query = $this->database->select('simple_like', 'sil');
    $query->addField('sil', 'id');
    $query->condition('sil.entity', $entity);
    $query->condition('sil.entity_id', $entity_id);
    $query->condition('sil.bundle', $bundle);
    $query->condition('sil.status', 1);
    return $query->countQuery()->execute()->fetchField();
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // We need this function, because interface requires it.
    // But nothing is needed here, it's all ajax above.
  }

  /**
   * Get current entity, if any.
   */
  private function getCurrentEntity() {
    foreach ($this->route->getParameters() as $param) {
      if ($param instanceof EntityInterface) {
        return $param;
      }
    }

    return NULL;
  }

}
