<?php

declare(strict_types=1);

namespace Drupal\Tests\simple_like_button\Functional;

use Drupal\block\Entity\Block;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\Tests\BrowserTestBase;

/**
 * Functional coverage for the Simple Like Button block.
 *
 * - Installs module
 * - Places the block on <front>
 * - Confirms the like button is visible for authenticated users
 */
final class LikeButtonBlockTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'block',
    'node',
    'simple_like_button',
  ];

  /**
   * Use a very simple core theme.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a basic 'page' content type.
    if (!NodeType::load('page')) {
      $type = NodeType::create([
        'type' => 'page',
        'name' => 'Basic page',
      ]);
      $type->save();
    }

    // Create a published node and make it the front page.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Front page node',
      'status' => 1,
    ]);
    $node->save();
    $front_path = '/node/' . $node->id();
    $this->config('system.site')
      ->set('page.front', $front_path)
      ->save();

    // Place the Simple Like Button block in the Stark "content" region.
    // Plugin ID is defined in the block annotation: id = "simple_like_button".
    Block::create([
      'id' => 'simple_like_button_test',
      'theme' => $this->config('system.theme')->get('default'),
      'plugin' => 'simple_like_button',
      'region' => 'content',
      'weight' => 0,
      'visibility' => [],
      'settings' => [],
    ])->save();
  }

  /**
   * Ensure the like button shows on the <front> page for authenticated users.
   */
  public function testLikeButtonVisibleOnFront(): void {
    // Authenticated user with permission to view content.
    $account = $this->drupalCreateUser([
      'access content',
    ]);
    $this->drupalLogin($account);

    // Visit <front>.
    $this->drupalGet('<front>');
    $this->assertSession()->statusCodeEquals(200);

    // The form only renders for authenticated users on entity routes.
    // Assert the Like form is present and the button appears.
    // The form ID is derived from ::getFormId() => 'like_form' => 'like-form' in markup.
    $this->assertSession()->elementExists('css', 'form#like-form');
    // Button value starts with "Like ·" or "Liked ·" depending on state.
    $this->assertSession()->elementExists('css', 'form#like-form #edit-actions');
  }

}
