<?php

namespace Drupal\simple_oauth_logout\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\simple_oauth_revoke\Controller\Oauth2Revoke;
use GuzzleHttp\Psr7\Response;
use League\OAuth2\Server\Exception\OAuthServerException;
use League\OAuth2\Server\ResourceServer;
use Psr\Http\Message\ResponseInterface;
use Symfony\Bridge\PsrHttpMessage\HttpMessageFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Handles OAuth logout by revoking tokens and destroying Drupal sessions.
 */
final class Oauth2Logout extends ControllerBase {

  /**
   * The OAuth2 revoke controller.
   *
   * @var \Drupal\simple_oauth_revoke\Controller\Oauth2Revoke
   */
  protected $revokeController;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The OAuth resource server.
   *
   * @var \League\OAuth2\Server\ResourceServer
   */
  protected $resourceServer;

  /**
   * The HTTP message factory.
   *
   * @var \Symfony\Bridge\PsrHttpMessage\HttpMessageFactoryInterface
   */
  protected $messageFactory;

  /**
   * Constructs a new Oauth2Logout object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\simple_oauth_revoke\Controller\Oauth2Revoke $revoke_controller
   *   The OAuth2 revoke controller.
   * @param \League\OAuth2\Server\ResourceServer $resource_server
   *   The OAuth resource server.
   * @param \Symfony\Bridge\PsrHttpMessage\HttpMessageFactoryInterface $message_factory
   *   The HTTP message factory.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    Connection $database,
    LoggerChannelFactoryInterface $logger_factory,
    ModuleHandlerInterface $module_handler,
    Oauth2Revoke $revoke_controller,
    ResourceServer $resource_server,
    HttpMessageFactoryInterface $message_factory,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->database = $database;
    $this->loggerFactory = $logger_factory;
    $this->moduleHandler = $module_handler;
    $this->revokeController = $revoke_controller;
    $this->resourceServer = $resource_server;
    $this->messageFactory = $message_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $revoke_controller = Oauth2Revoke::create($container);
    return new static(
      $container->get('entity_type.manager'),
      $container->get('database'),
      $container->get('logger.factory'),
      $container->get('module_handler'),
      $revoke_controller,
      $container->get('simple_oauth.server.resource_server.factory')->get(),
      $container->get('psr7.http_message_factory'),
    );
  }

  /**
   * Processes POST requests to /oauth/logout.
   *
   * This endpoint accepts the same parameters as /oauth/revoke,
   * revokes the tokens, and destroys all Drupal sessions for the user.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The logout request object.
   *
   * @return \Psr\Http\Message\ResponseInterface
   *   The response.
   */
  public function logout(Request $request): ResponseInterface {
    $body = $request->getContent();
    $params = [];
    parse_str($body, $params);

    $token = $params['token'] ?? NULL;

    if (!$token) {
      $error_response = json_encode(['error' => 'invalid_request', 'message' => 'Missing token parameter']);
      return new Response(
        400,
        ['Content-Type' => 'application/json'],
        $error_response ?: ''
      );
    }

    $user_id = NULL;
    $sessions_destroyed = 0;

    try {
      // Use the same approach as simple_oauth_revoke to validate the token
      // and get the user ID from it.
      $fake_request = clone $request;
      $token_string = is_array($token) ? implode('', $token) : (string) $token;
      $fake_request->headers->set('Authorization', "Bearer " . $token_string);

      try {
        $authenticated_request = $this->resourceServer->validateAuthenticatedRequest($this->messageFactory->createRequest($fake_request));
        $attributes = $authenticated_request->getAttributes();

        if (isset($attributes['oauth_user_id'])) {
          $user_id = $attributes['oauth_user_id'];
          $this->loggerFactory->get('simple_oauth_logout')->info('Found user ID from token: @uid', ['@uid' => $user_id]);
        }
      }
      catch (OAuthServerException $e) {
        $this->loggerFactory->get('simple_oauth_logout')->warning('Failed to validate access token: @message', ['@message' => $e->getMessage()]);
      }

      // Now revoke the token using the simple_oauth_revoke module.
      // Call the revoke controller to handle token revocation.
      $this->revokeController->revoke($request);

      // If we found the user, destroy all their sessions.
      if ($user_id) {
        // Delete all sessions for this user.
        $sessions_destroyed = $this->database->delete('sessions')
          ->condition('uid', $user_id)
          ->execute();

        $this->loggerFactory->get('simple_oauth_logout')->info('Destroyed @count sessions for user @uid', [
          '@count' => $sessions_destroyed,
          '@uid' => $user_id,
        ]);

        // If we deleted sessions, invoke logout hooks.
        if ($sessions_destroyed > 0) {
          $account = $this->entityTypeManager->getStorage('user')->load($user_id);
          if ($account) {
            $this->moduleHandler->invokeAll('user_logout', [$account]);
          }
        }
      }
      else {
        $this->loggerFactory->get('simple_oauth_logout')->warning('No user ID found for token');
      }
    }
    catch (\Exception $e) {
      // Log the error but continue.
      $this->loggerFactory->get('simple_oauth_logout')->error('Error during logout: @message', [
        '@message' => $e->getMessage(),
      ]);
    }

    // Create response.
    $response_body = json_encode([
      'status' => 'success',
      'sessions_destroyed' => $sessions_destroyed,
    ]);

    $response = new Response(
      200,
      ['Content-Type' => 'application/json'],
      $response_body ?: ''
    );

    return $response;
  }

}
