<?php

namespace Drupal\simple_otp\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Datetime\DateFormatterInterface;


/**
 * Service for OTP generation and validation.
 */
class SimpleOtpService {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * The logger channel factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface;
   */
  protected $loggerFactory;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * Constructs a SimpleOtpService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    Connection $database,
    TimeInterface $time,
    LoggerChannelFactoryInterface $logger_factory,
    DateFormatterInterface $date_formatter
  ) {
    $this->configFactory = $config_factory;
    $this->database = $database;
    $this->time = $time;
    $this->loggerFactory = $logger_factory->get('simple_otp');
    $this->dateFormatter = $date_formatter;
  }

  /**
   * Generates an OTP for the given identifier.
   *
   * @param string $identifier
   *   The identifier (e.g., otp to email or phone number).
   *
   * @return string
   *   The generated OTP.
   */
  public function generateOtp(string $identifier, string $otp_to): string {
    // Check identifier and otp_to values whether empty or not.
    if (empty($identifier) && (in_array(strtolower($identifier), ['email', 'phone_number'])) && empty($otp_to)) {
        return 'failed';
    } else { 
      try {
        $config = $this->configFactory->get('simple_otp.settings');
        $length = $config->get('otp_length');
        $identifier = strtolower($identifier);
        $current_time = $this->time->getCurrentTime();
        $current_date_time = $this->dateFormatter->format($current_time, 'custom', 'Y-m-d H:i:s');
        // Generate random OTP.
        $otp = str_pad((string) random_int(0, pow(10, $length) - 1), $length, '0', STR_PAD_LEFT);
        
        $query = $this->database->select('simple_otp', 'so')
        ->fields('so', ['id'])
        ->condition('so.otp_identifier', $identifier)
        ->condition('so.otp_to', $otp_to)
        ->range(0, 1); // Limit to one result
        $result = $query->execute();
        // Fetch the result as an object
        $otp_object = $result->fetchObject();
        if (!empty($otp_object) && $otp_object->id > 0) {
            $data_to_update = [
              'otp' => $otp,
              'created_timestamp' => $current_date_time,
            ];
            // Build and execute the update query.
            $num_updated = $this->database->update('simple_otp')
              ->fields($data_to_update)
              ->condition('id', $otp_object->id)
              ->execute();

          // Check the number of affected rows.
          if ($num_updated > 0) {
            return $otp;
          } 
        } else {
          $data_to_insert = [
            'otp_identifier' => $identifier,
            'otp_to' => $otp_to,
            'otp' => $otp,
            'created_timestamp' => $current_date_time,
          ];
          // Add new OTP database
          $this->database->insert('simple_otp')
          ->fields($data_to_insert)
          ->execute();
          return $otp;
        }
      } catch (\Exception $e) {
        // Handle any other generic exceptions.
        $this->loggerFactory->error('Generic exception caught: ' . $e->getMessage());
        return 'failed';
      } 
    } 
    return 'failed';
  }

  /**
   * Validates an OTP for the given identifier.
   *
   * @param string $identifier
   *   The identifier (e.g., otp for email or phone number).
   * @param string $otp_to
   *   The otp_to (e.g., email or phone number).
   * @param string $otp
   *   The OTP to validate.
   *
   * @return bool
   *   TRUE if valid, FALSE otherwise.
   */
  public function validateOtp(string $identifier, string $otp_to, string $otp): bool {
    // Check identifier and otp values whether empty or not.
    if (!empty($identifier) && (in_array(strtolower($identifier), ['email', 'phone_number'])) && !empty($otp_to) && !empty($otp)) {
      try { 
        $config = $this->configFactory->get('simple_otp.settings');
        $expiry_minutes = $config->get('expiry_time');
        if (!empty($expiry_minutes) && $expiry_minutes > 0) {
          $current_time = $this->time->getCurrentTime();
          $before_time = $current_time - ($expiry_minutes*60);
          $current_date_time = $this->dateFormatter->format($current_time, 'custom', 'Y-m-d H:i:s');
          $before_date_time = $this->dateFormatter->format($before_time, 'custom', 'Y-m-d H:i:s');

          $query = $this->database->select('simple_otp', 'so')
          ->fields('so', ['id'])
          ->condition('so.otp_identifier', $identifier)
          ->condition('so.otp_to', $otp_to)
          ->condition('so.otp', $otp)
          ->condition('so.created_timestamp', $before_date_time, '>=')
          ->condition('so.created_timestamp', $current_date_time, '<=')
          ->range(0, 1); // Limit to one result
          $result = $query->execute();
          // Fetch the result as an object if OTP is valid.
          $otp_object = $result->fetchObject();
          if (!empty($otp_object) && $otp_object->id > 0) {
            return TRUE;
          } else {
            return FALSE;
          } 
        } else {
          return FALSE;
        }
      } catch (\Exception $e) {
        // Handle any other generic exceptions.
        $this->loggerFactory->error('Generic exception caught: ' . $e->getMessage());
        return FALSE;
      } 
    }
    return FALSE;
  }
}