<?php

namespace Drupal\simple_search_block\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\path_alias\AliasManagerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a simple search form.
 */
class SimpleSearchForm extends FormBase {
  /**
   * The path alias manager service.
   *
   * @var \Drupal\path_alias\AliasManagerInterface
   */
  protected $aliasManager;

  /**
   * The current request.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected $request;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'simple_search_block_form';
  }

  /**
   * Constructs a Simple Search Form.
   *
   * @param \Drupal\path_alias\AliasManagerInterface $path_alias_manager
   *   The path alias manager service.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   */
  public function __construct(AliasManagerInterface $path_alias_manager, Request $request) {
    $this->aliasManager = $path_alias_manager;
    $this->request = $request;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('path_alias.manager'),
      $container->get('request_stack')->getCurrentRequest()
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, string $search_page_route = '', string $search_page_filter = '', string $search_filter_title = '', string $search_filter_placeholder = '') {
    $form_state
      ->setMethod('GET')
      ->setAlwaysProcess(TRUE)
      ->setCached(FALSE)
      ->disableRedirect();

    $form['#id'] = 'simple-search-block-form';
    $form['#attributes'] = [
      'role' => 'search',
      'class' => ['simple-search-block-form'],
    ];

    $url = Url::fromRoute($search_page_route)->toString();
    $pathAlias = $this->aliasManager->getAliasByPath($url);

    // Get route to View results page.
    $form['#action'] = $pathAlias;
    $filter = $this->request->query->get($search_page_filter);
    $form[$search_page_filter] = [
      '#type' => 'search',
      '#title' => $this->t('%title', ['%title' => $search_filter_title]),
      '#title_display' => 'invisible',
      '#placeholder' => $this->t('%placeholder', ['%placeholder' => $search_filter_placeholder]),
      '#required' => FALSE,
      '#default_value' => !empty($filter) ? strip_tags(urldecode($filter)) : '',
    ];
    $form['submit_container'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['simple-search-block-submit']],
    ];
    $form['submit_container']['submit'] = [
      '#id' => 'edit-submit-simple-search-block',
      '#type' => 'submit',
      '#value' => $this->t('Search'),
    ];

    $form['#after_build'][] = [get_class($this), 'afterBuild'];

    // Disable form caching.
    $form['#cache'] = ['max-age' => 0];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Do nothing.
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Do nothing.
  }

  /**
   * After build callback.
   */
  public static function afterBuild(array $form, FormStateInterface $form_state) {
    unset($form['form_token']);
    unset($form['form_build_id']);
    unset($form['form_id']);
    return $form;
  }

}
