<?php

namespace Drupal\simple_search_block\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Routing\RouteProviderInterface;
use Drupal\Core\Block\Attribute\Block;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Provides a simple search block.
 */
#[Block(
  id: 'simple_search_block',
  admin_label: new TranslatableMarkup('Simple Search Block'),
  category: new TranslatableMarkup('Search'),
)]
class SimpleSearchBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * Drupal\Core\Form\FormBuilderInterface definition.
   *
   * @var formBuilder
   */
  protected $formBuilder;

  /**
   * The route provider service.
   *
   * @var \Drupal\Core\Routing\RouteProviderInterface
   */
  protected $routeProvider;

  /**
   * Constructs a Simple Search Form Block.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder service.
   * @param \Drupal\Core\Routing\RouteProviderInterface $route_provider
   *   The route provider service.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    FormBuilderInterface $form_builder,
    RouteProviderInterface $route_provider,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->formBuilder = $form_builder;
    $this->routeProvider = $route_provider;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'search_page_route' => '',
      'search_page_filter' => '',
      'search_filter_title' => 'Search',
      'search_filter_placeholder' => 'Enter keywords or phrases',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form['search_page_route'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search page route'),
      '#description' => $this->t('The route to submit the form to.'),
      '#default_value' => $this->configuration['search_page_route'],
      '#required' => TRUE,
    ];

    $form['search_page_filter'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search page filter'),
      '#description' => $this->t('The filter to use for the search query string.'),
      '#default_value' => $this->configuration['search_page_filter'],
      '#required' => TRUE,
    ];

    $form['search_filter_title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search filter title'),
      '#description' => $this->t('The title for the search input field.'),
      '#default_value' => $this->configuration['search_filter_title'],
      '#required' => TRUE,
    ];

    $form['search_filter_placeholder'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search filter placeholder'),
      '#description' => $this->t('The placeholder text for the search input field.'),
      '#default_value' => $this->configuration['search_filter_placeholder'],
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    $values = $form_state->getValues();
    $this->configuration['search_page_route'] = $values['search_page_route'];
    $this->configuration['search_page_filter'] = $values['search_page_filter'];
    $this->configuration['search_filter_title'] = $values['search_filter_title'];
    $this->configuration['search_filter_placeholder'] = $values['search_filter_placeholder'];
  }

  /**
   * {@inheritdoc}
   */
  public function blockValidate($form, FormStateInterface $form_state) {
    $route = $form_state->getValue('search_page_route');

    // Use Drupal's route system to check if the route exists.
    try {
      $this->routeProvider->getRouteByName($route);
    }
    catch (\Exception $e) {
      $form_state->setErrorByName('search_page_route', $this->t('The route %route does not exist.', ['%route' => $route]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('form_builder'),
      $container->get('router.route_provider'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $build['content'] = $this->formBuilder->getForm('Drupal\simple_search_block\Form\SimpleSearchForm', $this->configuration['search_page_route'], $this->configuration['search_page_filter'], $this->configuration['search_filter_title'], $this->configuration['search_filter_placeholder']);
    // Add classes to the block.
    $build['#attributes']['class'][] = 'simple-search-block';

    return $build;
  }

}
