<?php

namespace Drupal\simple_sitemap_diwoo;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\media\MediaInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Klasse om een XSD complexType te mappen naar Drupal Form API elementen.
 */
class DiwooXsdParser implements ContainerInjectionInterface {
  use StringTranslationTrait;

  protected const PREFIX = 'diwoo:';
  protected const MEDIA_BUNDLES = ['document_bundle'];

  /**
   * The configuration object for the module.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $config;

  /**
   * The media storage handler.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $mediaStorage;

  /**
   * The path to the XSD file for fields.
   *
   * @var string
   */
  protected string $xsdFieldsPath;

  /**
   * The path to the XSD file for lists.
   *
   * @var string
   */
  protected string $xsdListsPath;

  /**
   * An array of complex types parsed from the XSD file.
   *
   * @var array
   */
  protected array $complexTypes;

  /**
   * An array of select lists with options retrieved from the xsd.
   *
   * @var array
   */
  protected array $xdsLists;

  /**
   * The field name in the form used for states.
   *
   * @var string
   */
  protected string $fieldName = '';

  /**
   * The required states for form fields based on enabled field.
   *
   * @var array
   */
  protected array $requiredStates = [];

  /**
   * Constructs a new XsdToDrupalForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $fileUrlGenerator
   *   The file URL generator service.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ModuleHandlerInterface $moduleHandler,
    protected FileUrlGeneratorInterface $fileUrlGenerator,
  ) {
    $this->config = $configFactory->get('simple_sitemap_diwoo.settings');
    $this->mediaStorage = $this->entityTypeManager->getStorage('media');

    $path = $this->moduleHandler->getModule('simple_sitemap_diwoo')->getPath() . '/assets/';
    $this->xsdFieldsPath = $path . 'diwoo-metadata.xsd';
    $this->xsdListsPath = $path . 'diwoo-metadata-lijsten.xsd';
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('module_handler'),
      $container->get('file_url_generator'),
    );
  }

  /**
   * Retrieves the parsed XSD lists.
   *
   * @return array
   *   An associative array containing the parsed lists and their options.
   */
  public function getXsdLists(): array {
    if (empty($this->xdsLists)) {
      $this->parseXsdLists();
    }
    return $this->xdsLists;
  }

  /**
   * Retrieves the options for a given XSD list.
   *
   * @param string $name
   *   The name of the XSD list.
   *
   * @return array
   *   An array of options for the specified XSD list.
   */
  protected function getXsdListOptions(string $name) {
    return $this->xdsLists[$name]['options'] ?? [];
  }

  /**
   * Sets the field name which is used in the form.
   *
   * By setting this value you enable the option to toggle the visibility of
   * non-required fields.
   *
   * @param string $name
   *   The name of the field to set.
   */
  public function setFormFieldName(string $name) {
    $this->fieldName = $name;
  }

  /**
   * Sets the required states for form fields.
   *
   * @param array $states
   *   An array of states to mark fields as required.
   */
  public function setRequiredStates(array $states) {
    $this->requiredStates = ['required' => $states];
  }

  /**
   * Builds a Drupal form based on the parsed XSD fields.
   *
   * @param array $values
   *   The default values for the form fields.
   *
   * @return array
   *   The structured form array.
   */
  public function buildForm(array $values): array {
    // Parse xsd files so we can easily use it in our form.
    $this->parseXsdFields();
    $this->parseXsdLists();

    // Okay, Let's try to build the form of DiWoo.
    $form = [
      '#tree' => TRUE,
      'intro' => [
        '#markup' => '<p>' . $this->complexTypes['DiWooType']['docs']['text'] . '</p>',
      ],
    ];

    // Show the token helper.
    $form['token_help'] = [
      '#theme' => 'token_tree_link',
      '#token_types' => ['media'],
      '#global_types' => FALSE,
      '#dialog' => TRUE,
    ];

    // Checkbox to hide non required fields.
    if ($this->fieldName) {
      $form['show_non_required'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Show all metadata fields'),
        '#description' => $this->t('The non-required fields are hidden by default'),
      ];
    }

    foreach ($this->complexTypes['DiWooType']['elements'] as $name => $data) {
      $this->renderXsdFormElement($name, $data, $form, $values);
    }

    return $form;
  }

  /**
   * Renders an XSD form element.
   *
   * @param string $name
   *   The name of the form element.
   * @param array $data
   *   The data associated with the form element.
   * @param array &$form
   *   The form array to which the element will be added.
   * @param array $values
   *   The form values.
   */
  protected function renderXsdFormElement(string $name, array $data, array &$form, array $values) {
    $camelcase_split = preg_replace('/([a-z])([A-Z])/', '$1 $2', $name);
    $title = ucfirst(strtolower($camelcase_split));

    // Check if it's a custom/complex field from xsd.
    $custom_type_name = str_replace(self::PREFIX, '', $data['type']);
    $custom_type = $this->complexTypes[$custom_type_name] ?? NULL;

    // Exception for a custom field.
    if ($custom_type_name === 'documentverwijzingType') {
      $custom_type['type'] = 'entity_reference';
    }

    // Check if it's a multiple field.
    $is_multiple_field = $data['max'] !== '1';

    // Children of custom types are only required if the parent is required.
    $required = $data['min'] ?? FALSE;
    if (isset($form['#_required']) && !$form['#_required']) {
      $required = FALSE;
    }

    // Build the states for the element. Required and/or visible.
    $states = [];
    if ($required) {
      $states = $this->requiredStates;
    }
    elseif ($this->fieldName) {
      // Hide the non required fields by default with a checkbox.
      $states = [
        'visible' => [
          ':input[name="' . $this->fieldName . '[show_non_required]"]' => ['checked' => TRUE],
        ],
      ];
    }

    // Get the values for the given fields.
    $default_value = $values[$name] ?? NULL;

    // The custom type can also be a variant of a field, so no need to dig.
    $type = $custom_type['type'] ?? $data['type'];
    switch ($type) :
      case 'xs:string':
        // For now an exception for a textarea. Can't read it from the xsd.
        $form[$name] = [
          '#type' => $name === 'omschrijving' ? 'textarea' : 'textfield',
          '#title' => $title,
          '#description' => $data['docs']['description'],
          '#states' => $states,
          '#default_value' => $default_value,
        ];
        break;

      case 'xs:date':
      case 'xs:dateTime':
        // There are date and datetime fields, but if we show them we can't add
        // our token which are very usefull for dates.
        $form_types = [
          'xs:date' => [
            'type' => 'date',
            'format' => 'Y-m-d',
            'example' => '2025-04-23',
            'token' => '[media:created:date:custom:Y-m-d]',
          ],
          'xs:dateTime' => [
            'type' => 'datetime',
            'format' => 'c',
            'example' => '2020-09-10T09:35:59+01:00',
            'token' => '[media:created:date:custom:c]',
          ],
        ];

        $form[$name] = [
          '#type' => 'textfield',
          '#title' => $title,
          '#placeholder' => $this->t('Example: @example', [
            '@example' => $form_types[$type]['example'],
          ]),
          '#description' => $data['docs']['description'],
          '#states' => $states,
          '#default_value' => $default_value,
        ];

        // Add token as description.
        $form[$name]['#description'] .= '<br/> - Date format: ' . $form_types[$type]['format'];
        $form[$name]['#description'] .= '<br/> - Value example: ' . $form_types[$type]['example'];
        $form[$name]['#description'] .= '<br/> - Token example: ' . $form_types[$type]['token'];
        break;

      case 'entity_reference':
        $entity_type = 'media';
        if ($entity_id = $default_value['_entity_id'] ?? NULL) {
          $storage = $this->entityTypeManager->getStorage($entity_type);
          $default_value = $storage->load($entity_id);
        }

        $form[$name]['_entity_id'] = [
          '#type' => 'entity_autocomplete',
          '#title' => $title,
          '#description' => $this->t('Type to search for a document to link'),
          '#target_type' => $entity_type,
          '#selection_settings' => [
            'target_bundles' => self::MEDIA_BUNDLES,
          ],
          '#default_value' => $default_value ?: NULL,
          '#states' => $states,
        ];
        break;

      case 'resource':
        $form[$name]['_list'] = [
          '#type' => 'hidden',
          '#value' => $custom_type['resource'],
        ];

        $form[$name]['_resource'] = [
          '#type' => 'select',
          '#title' => $title,
          '#description' => $data['docs']['description'],
          '#options' => $this->getXsdListOptions($custom_type['resource']),
          '#empty_option' => $this->t('- Select -'),
          '#states' => $states,
          '#default_value' => $default_value['_resource'] ?? NULL,
        ];

        break;

      default:
        if ($custom_type) {
          // Custom type with cildren.
          $form[$name] = [
            '#type' => 'details',
            '#title' => $title,
            '#states' => $states,
            '#_required' => $required,
          ];
          foreach ($custom_type['elements'] as $child_name => $child_data) {
            $this->renderXsdFormElement($child_name, $child_data, $form[$name], $default_value ?? []);
          }
        }
        else {
          $form[$name] = [
            '#markup' => $name . ' NOT IMPLEMENTED WITH TYPE ' . $data['type'],
          ];
        }
        break;
    endswitch;

    if ($is_multiple_field) {
      $this->updateToMulitpleFields($form, $name, $default_value);
    }
  }

  /**
   * Updates a form element to support multiple fields.
   *
   * @param array &$form
   *   The form array to be updated.
   * @param string $name
   *   The name of the form element to be updated.
   * @param mixed $default_value
   *   The default value(s) for the form element.
   */
  protected function updateToMulitpleFields(array &$form, $name, $default_value) {
    $field = $form[$name];
    $wrapper_id = 'multiple-field-wrapper-' . $name;
    $form[$name] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => $wrapper_id,
      ],
    ];

    // Check how many values we have +1.
    $default_value = array_values(array_filter($default_value ?: []));
    $num_fields = count($default_value) + 1;
    for ($i = 0; $i < $num_fields; $i++) {
      $form[$name][$i] = $field;
      $this->updateMultipleDefaults($form[$name][$i], $i, $default_value[$i] ?? NULL);
    }

    // Add button.
    $class = get_class($this);
    $form[$name . '_add_more'] = [
      '#name' => $name . '_add_more',
      '#type' => 'submit',
      '#value' => (string) $this->t('Add field'),
      '#suffix' => '<small>' . ((string) $this->t('(empty fields will be deleted automatically)')) . '</small>',
      '#submit' => [[$class, 'addExtraMultipleField']],
      '#ajax' => [
        'callback' => [$class, 'ajaxRefreshMultipleField'],
        'wrapper' => $wrapper_id,
      ],
      '#_multiple_field' => $name,
    ];
  }

  /**
   * Updates default values for multiple fields in a form element.
   *
   * @param array &$field
   *   The form field array to update.
   * @param int $i
   *   The index of the current field in the multiple field set.
   * @param mixed $default_value
   *   The default value to set for the field.
   */
  protected function updateMultipleDefaults(array &$field, $i, $default_value) {
    // Unset states for required on multiple fields.
    if ($i) {
      $field['#states'] = [];
    }

    // Update the default value.
    if (isset($field['_entity_id'])) {
      // Turn into entity.
      if (!empty($default_value['_entity_id'])) {
        $storage = $this->entityTypeManager->getStorage($default_value['_entity_type']);
        $default_value = $storage->load($default_value['_entity_id']);
        $field['_entity_id']['#default_value'] = $default_value;
      }
    }
    elseif ($children = Element::children($field)) {
      // Loop through children fields (container fields).
      foreach ($children as $child) {
        $this->updateMultipleDefaults($field[$child], $i, $default_value[$child] ?? NULL);
      }
    }
    else {
      $field['#default_value'] = $default_value;
    }
  }

  /**
   * Ajax submit handler for add more button, just to rebuild the form.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public static function addExtraMultipleField(array &$form, FormStateInterface $form_state) {
    $form_state->setRebuild(TRUE);
  }

  /**
   * Ajax callback to refresh the multiple field container.
   *
   * @param array $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   The updated form container for the multiple field.
   */
  public static function ajaxRefreshMultipleField(array &$form, FormStateInterface $form_state) {
    $trigger = $form_state->getTriggeringElement();

    // Replace last item (add more button) for the wrapper name.
    $parents = $trigger['#parents'];
    $parents = array_slice($parents, 0, -1);
    $parents[] = $trigger['#_multiple_field'];

    $container = $form;
    foreach ($parents as $key) {
      // If form used in entity_form the widget is used.
      if (is_numeric($key)) {
        $container = $container['widget'];
      }
      $container = $container[$key];
    }
    return $container;
  }

  /**
   * Parses the XSD file for fields and registers complex types.
   */
  protected function parseXsdFields() {
    $xml = simplexml_load_file($this->xsdFieldsPath);

    // Register all complex types.
    foreach ($xml->xpath('//xs:complexType') as $type) {
      $name = (string) $type['name'];
      $this->complexTypes[$name] = [
        '_raw' => $type,
        'docs' => $this->getDocsOfElement($type),
      ];

      // Get elements via sequence or all.
      $children = $type->xpath('xs:sequence');
      if (!$children) {
        $children = $type->xpath('xs:all');
      }
      foreach ($children as $sequence) {
        foreach ($sequence->xpath('xs:element') as $element) {
          $element_name = (string) $element['name'];
          $element_type = (string) $element['type'] ?? '';
          $this->complexTypes[$name]['elements'][$element_name] = [
            'type' => $element_type,
            'min' => !!(int) $element['minOccurs'],
            'max' => (string) $element['maxOccurs'],
            'docs' => $this->getDocsOfElement($element),
          ];
        }
      }

      // Define the complex types for form fields.
      foreach ($type->xpath('xs:simpleContent/xs:extension') as $extension) {
        $base = (string) $extension['base'] ?? NULL;
        if ($base === self::PREFIX . 'label') {
          $this->complexTypes[$name]['type'] = 'resource';

          // Add docs.
          if ($resource_docs = $extension->xpath('xs:attribute/xs:annotation/xs:documentation')[0] ?? NULL) {
            $this->complexTypes[$name]['docs']['text'] .= (string) $resource_docs;
          }

          // Add restriction/source.
          if ($restriction = $extension->xpath('xs:attribute/xs:simpleType/xs:restriction')[0] ?? NULL) {
            $this->complexTypes[$name]['resource'] = str_replace(self::PREFIX, '', (string) $restriction['base']);
          }
        }
      }
    }
  }

  /**
   * Parses the XSD file for lists and retrieves structured data.
   */
  protected function parseXsdLists() {
    $xml = simplexml_load_file($this->xsdListsPath);

    // Get simple types with a restriction and base.
    $lists = [];
    $inner_lists = [];
    $simpleTypes = $xml->xpath('//xs:simpleType[xs:restriction[@base]]');
    foreach ($simpleTypes as $type) {
      $name = (string) $type['name'];
      $restriction = $type->xpath('xs:restriction')[0];
      $base = (string) $restriction['base'];

      if (!$name) {
        continue;
      }

      $lists[$name] = [
        'name' => $name,
        'restriction' => $restriction,
        'base' => $base,
        'type' => $type,
        'docs' => $this->getDocsOfElement($type),
        'options' => [],
      ];

      // Get the lists which have simple type as base.
      $inner_lists[$name] = $base;
    }

    // We are now getting the options of the lists.
    foreach ($lists as $name => $type) {
      if ($type['base'] === 'xs:string') {
        foreach ($type['restriction']->xpath('xs:enumeration') as $enum) {
          $value = (string) $enum['value'];
          $docs = $this->getDocsOfElement($enum);
          $lists[$name]['options'][$value] = $docs['label'];
        }

        // Let's check if this list is used in other lists as simpleType.
        // Example; cww_plooi_filetypes is used in formatlist.
        $matches = array_filter($lists, function ($list) use ($name) {
          return isset($list['base']) && $list['base'] === self::PREFIX . $name;
        });

        // It's a sublist so no need to keep it in the full list.
        if ($matches) {
          foreach ($matches as $match) {
            // Set the docs.
            $lists[$match['name']]['docs'] = $lists[$name]['docs'];

            // Set the options.
            $lists[$match['name']]['options'] = $lists[$name]['options'];
          }

          unset($lists[$name]);
        }
      }
    }

    // In the xsd there is also an grouped list with union, let's fill those.
    foreach ($xml->xpath('//xs:simpleType[xs:union]') as $type) {
      $name = (string) $type['name'];
      $lists[$name] = [
        'name' => $name,
        'type' => 'union',
        'docs' => $this->getDocsOfElement($type),
        'options' => [],
      ];

      // Options are filled with other lists.
      foreach ($type->xpath('xs:union/xs:simpleType') as $element) {
        $restriction = $element->xpath('xs:restriction')[0];
        $sub_list_name = str_replace(self::PREFIX, '', (string) $restriction['base']);

        if ($sub_list = $lists[$sub_list_name]) {
          $lists[$name]['options'][$sub_list_name] = $sub_list['options'];
          unset($lists[$sub_list_name]);
        }
      }
    }

    $this->xdsLists = $lists;
  }

  /**
   * Retrieves documentation details from an XML element.
   *
   * @param \SimpleXMLElement $element
   *   The XML element from which to extract documentation.
   *
   * @return array
   *   An associative array containing 'source', 'text', and 'label' keys.
   */
  protected function getDocsOfElement(\SimpleXMLElement $element) {
    $docs = [
      'source' => '',
      'text' => '',
      'label' => '',
      'description' => '',
    ];

    if ($element = $element->xpath('xs:annotation/xs:documentation')[0] ?? NULL) {
      $docs['source'] = (string) $element['source'] ?? '';

      // Check if docs contains p html-tags.
      foreach ($element->children() as $child) {
        if (empty($docs['label'])) {
          // Trim p tag.
          $docs['label'] = strip_tags($child->asXML());
        }
        $docs['text'] .= $child->asXML();
      }

      // If still empty the element itself could be a string.
      if (empty($docs['text'])) {
        $docs['text'] = (string) $element;
        $docs['label'] = (string) $element;
      }
    }

    // Generate form field description.
    $docs['description'] = $docs['text'];
    if ($docs['source']) {
      $source = 'source: <a href="' . $docs['source'] . '" target="_blank">' . $docs['source'] . '</a>';
      $docs['description'] .= ($docs['description'] ? '<br/>' : '') . $source;
    }

    return $docs;
  }

  /**
   * {@inheritdoc}
   */
  public function massageMetadataValue(array $value, string $file_field_name = ''): array {
    $this->parseXsdLists();
    $this->flattenValueArray($value, $file_field_name);
    return $value;
  }

  /**
   * Flattens a nested array of values, converting specific objects to strings.
   *
   * @param array $value
   *   The array of values to be flattened. Nested arrays will be recursively
   *   processed, and empty values will be removed.
   * @param string $file_field_name
   *   The relative file field name to retrieve data from.
   *
   * @return void
   *   This method does not return any value.
   */
  protected function flattenValueArray(array &$value, string $file_field_name): void {
    // Don't save empty resource values.
    if (isset($value['_list'])) {
      if (empty($value['_resource'])) {
        $value = NULL;
        return;
      }

      // Get the label of the given value (save as resource for the sitemap).
      $options = $this->xdsLists[$value['_list']]['options'] ?? [];
      $value = [
        '_list' => $value['_list'],
        '_resource' => $value['_resource'],
        '_value' => $this->findNestedOption($options, $value['_resource']),
      ];
      return;
    }

    // Save entity reference as entity id, url and title.
    if (isset($value['_entity_id'])) {
      if (empty($value['_entity_id'])) {
        $value = NULL;
        return;
      }

      if ($entity = $this->mediaStorage->load($value['_entity_id'])) {
        $value = [
          '_entity_id' => $value['_entity_id'],
          '_entity_type' => $entity->getEntityTypeId(),
          '_resource' => $this->getMediaDocumentUrl($entity, $file_field_name),
          '_value' => $entity->label(),
        ];
        return;
      }

      // Entity not found, so nothing to save.
      $value = NULL;
      return;
    }

    // Let's clean up the nested array.
    foreach ($value as $key => &$sub_value) {
      if (is_array($sub_value)) {
        $this->flattenValueArray($sub_value, $file_field_name);
      }
      elseif ($sub_value instanceof DrupalDateTime) {
        $sub_value = $sub_value->format('Y-m-d\TH:i:sP');
      }

      // Delete fields ending with "_add_more" ajax-buttons.
      if (substr($key, -9) === '_add_more') {
        unset($value[$key]);
      }

      // Clear empty arrays or values.
      if (empty($sub_value)) {
        unset($value[$key]);
        continue;
      }
    }
  }

  /**
   * Generates the absolute URL for the media document.
   *
   * @param \Drupal\media\MediaInterface $media
   *   The media entity for which the document URL is generated.
   * @param string $file_field_name
   *   The relative file field name to retrieve data from.
   *
   * @return string
   *   The absolute URL of the media document, or an empty string if not found.
   */
  public function getMediaDocumentUrl(MediaInterface $media, string $file_field_name): string {
    if ($media->hasField($file_field_name)) {
      if ($file = $media->get($file_field_name)->entity ?? NULL) {
        return $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri());
      }
    }
    return '';
  }

  /**
   * Finds a nested option in a multidimensional array by its key.
   *
   * @param array $options
   *   The array of options to search through.
   * @param string $search_key
   *   The key to search for in the array.
   *
   * @return mixed|null
   *   The value of the found key, or NULL if the key does not exist.
   */
  protected function findNestedOption(array $options, $search_key) {
    foreach ($options as $key => $value) {
      if ($key === $search_key) {
        return $value;
      }
      if (is_array($value)) {
        if ($found = $value[$search_key] ?? NULL) {
          return $found;
        }
      }
    }
    return NULL;
  }

  /**
   * Sets default values for the provided array based on configuration.
   *
   * @param array $values
   *   The array of values to set defaults for.
   */
  public function setDefaultValues(array &$values) {
    $defaults = $this->config->get('meta_defaults') ?: [];
    $this->overrideDefaultValue($values, $defaults);
  }

  /**
   * Overrides default values in the provided array with given defaults.
   *
   * @param array &$values
   *   The array of values to be updated with defaults.
   * @param array $default_values
   *   The array of default values to apply.
   */
  protected function overrideDefaultValue(array &$values, $default_values) {
    foreach ($default_values as $key => $default_value) {
      // No value for the given default array. Just override.
      if (empty($values[$key])) {
        // Ignore numbers (multiple field).
        if (!is_numeric($key)) {
          $values[$key] = $default_value;
        }
      }
      elseif (is_array($default_value)) {
        // We need to dig deeper to check if the nested value is set.
        $this->overrideDefaultValue($values[$key], $default_value);
      }
      else {
        // Just a field but filled with a value so we just leave it.
      }
    }
  }

}
