<?php

namespace Drupal\simple_sitemap_diwoo\Form;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\simple_sitemap_diwoo\DiwooXsdParser;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides form to manage settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * Constructs a SettingsForm object.
   *
   * @param \Drupal\simple_sitemap_diwoo\DiwooXsdParser $diwooXsdParser
   *   The DiwooXsdParser service.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entityTypeBundleInfo
   *   The entity type bundle info service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   */
  public function __construct(
    protected DiwooXsdParser $diwooXsdParser,
    protected EntityFieldManagerInterface $entityFieldManager,
    protected EntityTypeBundleInfoInterface $entityTypeBundleInfo,
    protected ModuleHandlerInterface $moduleHandler,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('simple_sitemap_diwoo.xsd_parser'),
      $container->get('entity_field.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['simple_sitemap_diwoo.settings'];
  }

  /**
   * Gets the configuration for extended sitemap index.
   *
   * @return \Drupal\Core\Config\Config
   *   The config.
   */
  protected function getEditableConfig() {
    return $this->config('simple_sitemap_diwoo.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'simple_sitemap_diwoo_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Settings'),
    ];

    $form['settings']['examples'] = [
      '#type' => 'html_tag',
      '#tag' => 'p',
      '#value' => $this->t('Examples and more information about DiWoo can be found at <a href="https://standaarden.overheid.nl/diwoo/metadata/diwoo-voorbeelden" target="_blank">https://standaarden.overheid.nl/diwoo/metadata/diwoo-voorbeelden</a>.'),
    ];

    $form['settings']['intro'] = [
      '#type' => 'html_tag',
      '#tag' => 'p',
      '#value' => $this->t('The DiWoo Sitemap includes media entities that have the required DiWoo metadata field completed. This metadata field can be added like any standard field and must be linked to an existing file field.'),
    ];

    $form['settings']['entities'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Entity type'),
        $this->t('Bundle'),
        $this->t('DiWoo meta field'),
        $this->t('File field'),
      ],
      '#rows' => self::getDiwooMetaFields(),
    ];

    // The metadata form which is shown on the entity edit forms.
    $form['metadata'] = [
      '#type' => 'details',
      '#title' => $this->t('Default value for the generated form based on the diwoo-metadata.xsd file'),
    ];

    // Set default values but keep the form state values.
    $values = $form_state->getValues()['metadata'] ?? [];
    if ($values) {
      // We need to clean up the form state values.
      $values = $this->diwooXsdParser->massageMetadataValue($values);
    }
    else {
      $this->diwooXsdParser->setDefaultValues($values);
    }
    $form['metadata'] += $this->diwooXsdParser->buildForm($values);

    // Show lists in table form.
    $form['diwoo_lists'] = [
      '#type' => 'details',
      '#title' => $this->t('Dynamic lists/resources from the diwoo-metadata-lijsten.xsd'),
    ];
    foreach ($this->diwooXsdParser->getXsdLists() as $key => $list) {
      $form['diwoo_lists'][$key]['select'] = [
        '#type' => 'select',
        '#title' => $key,
        '#empty_option' => $this->t('- Example select -'),
        '#options' => $list['options'],
      ];

      $rows = $this->optionsToTableRows($list['options']);
      $form['diwoo_lists'][$key]['table'] = [
        '#type' => 'details',
        '#open' => FALSE,
        '#title' => $this->t('Full list of @name (@count)', [
          '@name' => $key,
          '@count' => count($rows),
        ]),
        '_table' => [
          '#type' => 'table',
          '#header' => [
            'key/resource',
            'value/label',
            'group',
          ],
          '#rows' => $rows,
        ],
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * Converts options into table rows.
   *
   * @param array $options
   *   An associative array of options where keys are identifiers and values
   *   are either strings or nested arrays.
   *
   * @return array
   *   A two-dimensional array representing table rows, where each row contains
   *   the key/resource, value/label, and group.
   */
  protected function optionsToTableRows(array $options): array {
    $rows = [];
    foreach ($options as $key => $value) {
      if (is_array($value)) {
        foreach ($value as $k => $v) {
          $rows[] = [$k, $v, $key];
        }
      }
      else {
        $rows[] = [$key, $value, '-'];
      }
    }
    return $rows;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {

    // Flatten the meta data.
    $values = $form_state->getValues();
    if (!empty($values['metadata'])) {
      $values['metadata'] = $this->diwooXsdParser->massageMetadataValue($values['metadata']);
    }

    // Only save the defaults which will be used on the diwoo meta fields.
    $config = $this->configFactory()->getEditable('simple_sitemap_diwoo.settings');
    $config->set('meta_defaults', $values['metadata']);
    $config->save();

    $this->messenger()->addStatus($this->t('The configuration options have been saved.'));
  }

  /**
   * Retrieves fields of type 'diwoo_meta' for media entities.
   *
   * @return array
   *   An array of fields with details such as bundle, name, and file field.
   */
  public static function getDiwooMetaFields() {
    $entity_type_bundle_info = \Drupal::service('entity_type.bundle.info');
    $entity_field_manager = \Drupal::service('entity_field.manager');
    $entity_type = 'media';

    $fields = [];
    $bundles = $entity_type_bundle_info->getBundleInfo($entity_type);
    foreach ($bundles as $bundle => $bundle_info) {
      $field_definitions = $entity_field_manager->getFieldDefinitions($entity_type, $bundle);

      foreach ($field_definitions as $field_name => $field_definition) {
        if ($field_definition->getType() === 'diwoo_meta') {
          $fields[] = [
            'entity_type' => $entity_type,
            'bundle' => $bundle,
            'name' => $field_name,
            'file_field' => $field_definition->getSetting('file_field'),
          ];
        }
      }
    }
    return $fields;
  }

}
