<?php

namespace Drupal\simple_sitemap_diwoo\Plugin\simple_sitemap\SitemapGenerator;

use Drupal\simple_sitemap\Plugin\simple_sitemap\SitemapGenerator\DefaultSitemapGenerator;

/**
 * Provides the DiWoo sitemap generator.
 *
 * @SitemapGenerator(
 *   id = "diwoo",
 *   label = @Translation("Diwoo sitemap generator"),
 *   description = @Translation("Generates a standard conform hreflang sitemap of your content including DiWoo mapping."),
 * )
 */
class DiwooSitemapGenerator extends DefaultSitemapGenerator {

  protected const XMLNS_DIWOO = 'https://standaarden.overheid.nl/diwoo/metadata/';
  protected const XMLNS_XSI = 'http://www.w3.org/2001/XMLSchema-instance';
  protected const XSI_SCHEMA_LOCATION =
    'https://www.sitemaps.org/schemas/sitemap/0.9 ' .
    'https://www.sitemaps.org/schemas/sitemap/0.9/sitemap.xsd ' .
    'https://standaarden.overheid.nl/diwoo/metadata/ ' .
    'https://standaarden.overheid.nl/diwoo/metadata/0.9.4/xsd/diwoo-metadata.xsd';

  /**
   * An array of attributes.
   *
   * @var array
   */
  protected const ATTRIBUTES = [
    'xmlns' => self::XMLNS,
    'xmlns:sitemap' => self::XMLNS,
    'xmlns:diwoo' => self::XMLNS_DIWOO,
    'xmlns:xsi' => self::XMLNS_XSI,
    'xsi:schemaLocation' => self::XSI_SCHEMA_LOCATION,
  ];

  /**
   * {@inheritdoc}
   */
  public function getChunkContent(array $links): string {
    // We keep it clean so now generated by or styling.
    $this->writer->openMemory();
    $this->writer->setIndent(TRUE);
    $this->writer->startSitemapDocument();

    $this->writer->startElement('urlset');
    $this->addSitemapAttributes();
    $this->addLinks($links);
    $this->writer->endElement();
    $this->writer->endDocument();

    return $this->writer->outputMemory();
  }

  /**
   * Adds attributes to the sitemap.
   */
  protected function addSitemapAttributes(): void {
    $attributes = self::ATTRIBUTES;
    $this->moduleHandler->alter('simple_sitemap_attributes', $attributes, $this->sitemap);
    foreach ($attributes as $name => $value) {
      $this->writer->writeAttribute($name, $value);
    }
  }

  /**
   * Adds URL elements to the sitemap.
   *
   * @param array $links
   *   An array of URL elements.
   */
  protected function addLinks(array $links): void {
    foreach ($links as $url_data) {
      $this->writer->startElement('url');
      $this->addUrl($url_data);
      $this->writer->endElement();
    }
  }

  /**
   * Adds a URL element to the sitemap.
   *
   * @param array $url_data
   *   An associative array containing URL properties and metadata.
   */
  protected function addUrl(array $url_data): void {
    parent::addUrl($url_data);

    // Add DiWoo attributes.
    $this->addDiwooAttributes($url_data);
  }

  /**
   * Converts an array to XML format.
   *
   * @param array $url_data
   *   An associative array containing URL properties and DiWoo metadata.
   */
  protected function addDiwooAttributes(array $url_data) {
    $data = $url_data['diwoo'] ?? [];

    // Create wrapper.
    $prefix = 'diwoo:';
    $this->writer->startElement($prefix . 'Document');
    $this->writer->startElement($prefix . 'DiWoo');

    // Add all children.
    foreach ($data as $name => $child) {
      $this->diwooArrayToXml($prefix, $name, $child);
    }

    $this->writer->endElement();
    $this->writer->endElement();
  }

  /**
   * Converts an array or value into XML format with a specified prefix.
   *
   * @param string $prefix
   *   The prefix to be added to the XML elements.
   * @param string $name
   *   The name of the XML element.
   * @param mixed $value
   *   The value or array to be converted into XML.
   */
  protected function diwooArrayToXml(string $prefix, string $name, $value) {
    if (!is_array($value)) {
      // Just a normal xml element.
      $this->writer->writeElement($prefix . $name, $value);
    }
    elseif (isset($value['_value'])) {
      // Save a xml element with resourece attribute.
      $this->writer->startElement($prefix . $name);
      $this->writer->writeAttribute('resource', $value['_resource']);
      $this->writer->text($value['_value']);
      $this->writer->endElement();
    }
    else {
      // This is a parent or multiple field.
      $is_multiple_field = array_keys($value) === range(0, count($value) - 1);

      // This is just a parent.
      if (!$is_multiple_field) {
        $this->writer->startElement($prefix . $name);
      }

      // Inner arrays can be numeric if it's a multiple field.
      foreach ($value as $key => $child) {
        $child_name = is_numeric($key) ? $name : $key;
        $this->diwooArrayToXml($prefix, $child_name, $child);
      }

      if (!$is_multiple_field) {
        $this->writer->endElement();
      }
    }
  }

  /**
   * Returns dummy data for testing purposes.
   */
  public static function getDummydata() {
    return [
      'publisher' => [
        '_resource' => 'https://identifier.overheid.nl/tooi/id/gemeente/gm9999',
        '_value' => 'gemeente Juinen',
      ],
      'identifiers' => [
        'identifier' => [
          'test',
          'test2',
        ],
      ],
      'titlecollectie' => [
        'officieleTitel' => 'Convenant busvervoer 2020 gemeente Juinen',
      ],
      'informatiecategorieen' => [
        'informatiecategorie' => [
          [
            '_resource' => 'x',
            '_value' => 'text',
          ],
          [
            '_resource' => 'y',
            '_value' => 'text',
          ],
        ],
      ],
      'documenthandelingen' => [
        'documenthandeling' => [
          [
            'soortHandeling' => [
              '_source' => 'resourcexx',
              '_value' => 'soort handeling',
            ],
            'atTime' => '01-01-2025 10:10:10',
          ],
        ],
      ],
    ];
  }

}
