<?php

namespace Drupal\simple_sitemap_diwoo\Plugin\simple_sitemap\UrlGenerator;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\file\FileInterface;
use Drupal\media\MediaInterface;
use Drupal\simple_sitemap\Exception\SkipElementException;
use Drupal\simple_sitemap\Logger;
use Drupal\simple_sitemap\Plugin\simple_sitemap\SimpleSitemapPluginBase;
use Drupal\simple_sitemap\Plugin\simple_sitemap\UrlGenerator\UrlGeneratorBase;
use Drupal\simple_sitemap\Settings;
use Drupal\simple_sitemap_diwoo\DiwooXsdParser;
use Drupal\simple_sitemap_diwoo\Form\SettingsForm;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Generates media entity URLs for DiWoo.
 *
 * @UrlGenerator(
 *   id = "diwoo_media",
 *   label = @Translation("DiWoo media URL generator"),
 *   description = @Translation("Generates URLs of media entities for DiWoo."),
 * )
 */
class DiwooMediaUrlGenerator extends UrlGeneratorBase {

  /**
   * Local cache for the available language objects.
   *
   * @var \Drupal\Core\Language\LanguageInterface[]
   */
  protected $languages;

  /**
   * Default language ID.
   *
   * @var string
   */
  protected $defaultLanguageId;

  /**
   * Entities per queue item.
   *
   * @var int
   */
  protected $entitiesPerDataset = 25;

  /**
   * EntityUrlGeneratorBase constructor.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\simple_sitemap\Logger $logger
   *   Simple XML Sitemap logger.
   * @param \Drupal\simple_sitemap\Settings $settings
   *   The simple_sitemap.settings service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\simple_sitemap_diwoo\DiwooXsdParser $diwooXsdParser
   *   The DiWoo XSD parser service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    Logger $logger,
    Settings $settings,
    LanguageManagerInterface $language_manager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected DiwooXsdParser $diwooXsdParser,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $logger, $settings);
    $this->languages = $language_manager->getLanguages();
    $this->defaultLanguageId = $language_manager->getDefaultLanguage()->getId();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): SimpleSitemapPluginBase {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('simple_sitemap.logger'),
      $container->get('simple_sitemap.settings'),
      $container->get('language_manager'),
      $container->get('entity_type.manager'),
      $container->get('simple_sitemap_diwoo.xsd_parser')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDataSets(): array {
    $data_sets = [];

    foreach (SettingsForm::getDiwooMetaFields() as $field) {
      $entity_type = 'media';

      $storage = $this->entityTypeManager->getStorage($entity_type);
      $query = $storage->getQuery()
        ->condition('bundle', $field['bundle'])
        ->condition('status', 1)
        ->condition($field['name'] . '.indexed', 1)
        ->accessCheck(FALSE);

      // Add the ids as batches.
      $data_set = [
        'entity_type' => $entity_type,
        'diwoo_field_name' => $field['name'],
        'file_field_name' => $field['file_field'],
        'id' => [],
      ];
      foreach ($query->execute() as $entity_id) {
        $data_set['id'][] = $entity_id;
        if (count($data_set['id']) >= $this->entitiesPerDataset) {
          $data_sets[] = $data_set;
          $data_set['id'] = [];
        }
      }

      // Add the last data set if there are some IDs gathered.
      if (!empty($data_set['id'])) {
        $data_sets[] = $data_set;
      }
    }

    return $data_sets;
  }

  /**
   * {@inheritdoc}
   */
  protected function processDataSet($data_set): array {
    /** @var \Drupal\Core\Entity\ContentEntityInterface $entity */
    foreach ($this->entityTypeManager->getStorage($data_set['entity_type'])->loadMultiple((array) $data_set['id']) as $entity) {
      try {
        $paths[] = $this->processEntity($entity, $data_set['diwoo_field_name'], $data_set['file_field_name']);
      }
      catch (SkipElementException $e) {
        continue;
      }
    }

    return $paths ?? [];
  }

  /**
   * Processes the given entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to process.
   * @param string $diwoo_field_name
   *   The name of the diwoo meta field.
   * @param string $file_field_name
   *   The name of the file field.
   *
   * @return array
   *   Processing result.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityMalformedException
   * @throws \Drupal\simple_sitemap\Exception\SkipElementException
   */
  protected function processEntity(ContentEntityInterface $entity, string $diwoo_field_name, string $file_field_name): array {
    if (!$entity->hasField($diwoo_field_name)) {
      throw new SkipElementException();
    }

    if (!$entity->hasField($file_field_name)) {
      throw new SkipElementException();
    }

    /** @var \Drupal\file\Entity\File $file */
    $file = $entity->get($file_field_name)->entity;
    if (!$file) {
      throw new SkipElementException();
    }

    // Add default sitemap data (don't need change freq or priority).
    $path_data = [
      'url' => $this->diwooXsdParser->getMediaDocumentUrl($entity, $file_field_name),
      'lastmod' => date('c', $file->getChangedTime()),
    ];

    /** @var \Drupal\simple_sitemap_diwoo\Plugin\Field\FieldType\DiwooMetaFieldItem $item */
    $item = $entity->get($diwoo_field_name)->first();
    $path_data['diwoo'] = $item->getDiwooMeta(TRUE);

    // Fill diwoo data based on entity and file.
    $path_data['diwoo'] += $this->autoFillDiwooData($entity, $file);

    return $path_data;
  }

  /**
   * Constructs DiWoo-specific metadata for a media entity.
   *
   * @todo Set auto fill for some fields based on the file or entity. We can add
   * the creation date, language or file type, but if we do we also have to
   * remove it from the form or check if it's not already filled.
   *
   * @param \Drupal\media\MediaInterface $entity
   *   The media entity being processed.
   * @param \Drupal\file\FileInterface $file
   *   The file associated with the media entity.
   *
   * @return array
   *   An array containing DiWoo metadata.
   */
  protected function autoFillDiwooData(MediaInterface $entity, FileInterface $file) {
    $data = [];

    // Add creation date (we are using the entity date).
    if (empty($data['creatiedatum'])) {
      $data['creatiedatum'] = date('Y-m-d', $file->getCreatedTime());
    }

    return $data;
  }

  /**
   * {@inheritdoc}
   */
  public function generate($data_set): array {
    try {
      return $this->processDataSet($data_set);
    }
    catch (SkipElementException $e) {
      return [];
    }
  }

}
