<?php

namespace Drupal\simple_sitemap_diwoo\Plugin\Field\FieldType;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Field\Attribute\FieldType;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\Core\Utility\Token;

/**
 * Plugin implementation of the 'diwoo_meta' field type.
 */
#[FieldType(
  id: "diwoo_meta",
  label: new TranslatableMarkup("DiWoo meta"),
  description: new TranslatableMarkup("A field with multiple DiWoo values for the sitemap."),
  weight: -50,
  default_widget: "diwoo_widget",
  default_formatter: "diwoo_formatter",
  cardinality: 1,
)]
class DiwooMetaFieldItem extends FieldItemBase {

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    return [
      'columns' => [
        'value' => [
          'type' => 'text',
          'size' => 'big',
        ],
        'indexed' => [
          'type' => 'int',
          'size' => 'tiny',
          'default' => 0,
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function mainPropertyName() {
    return 'indexed';
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    $properties['value'] = DataDefinition::create('diwoo_meta')
      ->setLabel(t('DiWoo meta'))
      ->setRequired(TRUE);

    $properties['indexed'] = DataDefinition::create('boolean')
      ->setLabel(t('Indexed'))
      ->setRequired(TRUE);

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings() {
    return [
      'file_field' => NULL,
    ] + parent::defaultFieldSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::fieldSettingsForm($form, $form_state);

    // Get the entity type and bundle for this field.
    $field_definition = $this->getFieldDefinition();
    $entity_type_id = $field_definition->getTargetEntityTypeId();
    $bundle = $field_definition->getTargetBundle();

    /** @var \Drupal\field\Entity\FieldConfig[] $fields */
    $fields = \Drupal::service('entity_field.manager')
      ->getFieldDefinitions($entity_type_id, $bundle);

    // Filter fields to include only file fields.
    $file_fields = [];
    foreach ($fields as $field_name => $field) {
      if ($field->getType() !== 'file') {
        continue;
      }
      if ($field->getFieldStorageDefinition()->getCardinality() === 1) {
        $file_fields[$field_name] = $field->getLabel() . ' (' . $field_name . ')';
      }
    }

    // Add a dropdown to select a file field.
    $form['file_field'] = [
      '#type' => 'select',
      '#title' => $this->t('File field for Metadata'),
      '#options' => $file_fields,
      '#default_value' => $this->getSetting('file_field') ?? NULL,
      '#description' => $this->t('Select a file field for which the metadata should be generated (the file field can only have on value).'),
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty() {
    $value = $this->get('value')->getValue();
    return $value === NULL || $value === '' || $value === Json::encode([]);
  }

  /**
   * Checks if the field is indexed.
   *
   * @return bool
   *   TRUE if the field is indexed, FALSE otherwise.
   */
  public function isIndexed() {
    return !!$this->get('indexed')->getValue();
  }

  /**
   * Retrieves the decoded DiWoo meta data.
   *
   * @return array
   *   An associative array of DiWoo metadata, or an empty array if none exists.
   */
  public function getDiwooMeta(bool $replace_tokens = FALSE) {
    $encoded = $this->value;
    if (empty($encoded)) {
      return [];
    }

    $metadata = Json::decode($encoded);
    if (!$replace_tokens) {
      return $metadata;
    }

    // If the default values are changed after an entity has been saved with the
    // meta we are missing some data. So let's add the defaults here as well.
    $parser = \Drupal::service('simple_sitemap_diwoo.xsd_parser');
    $parser->setDefaultValues($metadata);

    /** @var \Drupal\Core\Utility\Token $token **/
    $token = \Drupal::service('token');
    $entity = $this->getEntity();
    $token_data = [$entity->getEntityTypeId() => $entity];
    return $this->replaceTokens($token, $token_data, $metadata);
  }

  /**
   * Replaces tokens in the given value recursively.
   *
   * @param \Drupal\Core\Utility\Token $token
   *   The token service used for replacing tokens.
   * @param array $token_data
   *   The token data array containing entity information.
   * @param mixed $value
   *   The value in which tokens need to be replaced. String or an array.
   *
   * @return mixed
   *   The value with tokens replaced.
   */
  protected function replaceTokens(Token $token, array $token_data, $value) {
    if (is_array($value)) {
      if (isset($value['_value'])) {
        return $value;
      }
      foreach ($value as $key => $_value) {
        $value[$key] = $this->replaceTokens($token, $token_data, $_value);
      }
    }
    else {
      return $token->replace($value, $token_data);
    }
    return $value;
  }

}
