<?php

namespace Drupal\simple_sitemap_diwoo\Plugin\Field\FieldWidget;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\simple_sitemap_diwoo\DiwooXsdParser;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'diwoo_widget' widget.
 */
#[FieldWidget(
  id: 'diwoo_widget',
  label: new TranslatableMarkup('Collapsible Diwoo Widget'),
  field_types: [
    'diwoo_meta',
  ],
  multiple_values: FALSE,
)]
class DiwooWidget extends WidgetBase implements ContainerFactoryPluginInterface {
  use StringTranslationTrait;

  /**
   * {@inheritDoc}
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    protected ConfigFactoryInterface $configFactory,
    protected DiwooXsdParser $diwooXsdParser,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('config.factory'),
      $container->get('simple_sitemap_diwoo.xsd_parser'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'sidebar' => TRUE,
      'use_details' => TRUE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $element['sidebar'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Place field in sidebar'),
      '#default_value' => $this->getSetting('sidebar'),
      '#description' => $this->t('If checked, the field will be placed in the sidebar on entity forms.'),
    ];
    $element['use_details'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Wrap the meta tags in a collapsed details container.'),
      '#default_value' => $this->getSetting('use_details'),
      '#description' => $this->t('If checked, the contents of the field will be placed inside a collapsed details container.'),
      '#states' => [
        'visible' => [
          'input[name$="[sidebar]"]' => ['checked' => FALSE],
        ],
      ],
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    if ($this->getSetting('sidebar')) {
      $summary['sidebar'] = $this->t('Use sidebar: Yes');
    }
    else {
      $summary['sidebar'] = $this->t('Use sidebar: No');

      if ($this->getSetting('use_details')) {
        $summary['use_details'] = $this->t('Use details container: Yes');
      }
      else {
        $summary['use_details'] = $this->t('Use details container: No');
      }
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    /** @var \Drupal\simple_sitemap_diwoo\Plugin\Field\FieldType\DiwooMetaFieldItem $item */
    $item = $items[$delta];

    // Retrieve the values for each metatag from the serialized array.
    $values = [];
    if (!empty($item->value)) {
      $values = Json::decode($item->value);
    }

    // Make sure that this variable is always an array.
    if (!is_array($values)) {
      $values = [];
    }

    // Get the form state values for ajax calls.
    if ($user_input = $form_state->getUserInput()) {
      $raw_metadata = $user_input[$items->getName()][$delta]['metadata'] ?? [];
      $file_field_name = $this->getFieldSettings()['file_field'] ?? '';
      $values = $this->diwooXsdParser->massageMetadataValue($raw_metadata, $file_field_name);
    }

    // Combine the given values with the default values.
    $this->diwooXsdParser->setDefaultValues($values);

    // Build the dynamic Metadata form based on xsd file.
    $element += $this->buildDiwooMetaForm($items, $delta, $values);

    // If the "sidebar" option was checked on the field widget, put the form
    // element into the form's "advanced" group. Otherwise, let it default to
    // the main field area.
    $sidebar = $this->getSetting('sidebar');
    if ($sidebar) {
      $element['#group'] = 'advanced';
    }

    // If the "use_details" option was not checked on the field widget, put the
    // form element into normal container. Otherwise, let it default to a
    // detail's container.
    $details = $this->getSetting('use_details');
    if (!$sidebar && !$details) {
      $element['metadata']['#type'] = 'fieldset';
    }

    return $element;
  }

  /**
   * Builds the Diwoo meta form for the field widget.
   *
   * @param \Drupal\Core\Field\FieldItemListInterface $items
   *   The field item list.
   * @param int $delta
   *   The delta of the current item.
   * @param array $values
   *   The values for the form elements.
   *
   * @return array
   *   The renderable array representing the form elements.
   */
  protected function buildDiwooMetaForm(FieldItemListInterface $items, $delta, array $values) {
    $file_field_name = $this->getFieldSettings()['file_field'] ?? '';

    // Add checkbox to enable the Diwoo meta form.
    $indexed = $items[$delta]->indexed ?? FALSE;
    $element['indexed'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Index document in Diwoo Sitemap'),
      '#description' => $this->t('By checking this field and filling in the required the metadata, this file will be included to the Diwoo sitemap if configured in the Simple Sitemap module. (the metadata is representive for the file @file)', [
        '@file' => $file_field_name,
      ]),
      '#default_value' => $indexed,
    ];

    $selector = $items->getName() . '[' . $delta . '][indexed]';
    $enabled_state = [':input[name="' . $selector . '"]' => ['checked' => TRUE]];
    $element['metadata'] = [
      '#type' => 'details',
      '#title' => 'Diwoo Metadata',
      '#states' => [
        'visible' => $enabled_state,
      ],
    ];

    // Set required state based on the indexed field.
    $this->diwooXsdParser->setRequiredStates($enabled_state);
    $this->diwooXsdParser->setFormFieldName($items->getName() . '[' . $delta . '][metadata]');
    $element['metadata'] += $this->diwooXsdParser->buildForm($values);

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    $file_field_name = $this->getFieldSettings()['file_field'] ?? '';

    foreach ($values as &$value) {
      // Flatten the values array to remove empty options.
      if ($meta = $value['metadata'] ?? NULL) {
        $meta = $this->diwooXsdParser->massageMetadataValue($meta, $file_field_name);
      }

      $value = [
        'value' => $meta ? Json::encode($meta) : NULL,
        'indexed' => $value['indexed'] ?? FALSE,
      ];
    }

    return $values;
  }

}
